<?php
/**
 * Created by PhpStorm.
 * User: alberto
 * Date: 25/09/20
 * Time: 10.03
 */

namespace Drop\Oms\Model\Stock;

use Drop\Oms\Helper\ConfigurationHelper;
use Magento\Catalog\Model\Product\Attribute\Source\Status as ProductStatus;
use Magento\Catalog\Model\Product\Type as ProductType;
use Psr\Log\LoggerInterface;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable as ConfigurableProduct;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogInventory\Helper\Stock;
use Magento\CatalogInventory\Api\StockRegistryInterface;

class ConfigurableAlignment
{
    protected $logger;
    protected $configuration;
    protected $productCollectionFactory;
    protected $stockFilter;
    protected $stockRegistry;

    /**
     * ConfigurableAlignment constructor.
     * @param LoggerInterface $logger
     * @param ConfigurationHelper $configurationHelper
     * @param CollectionFactory $productCollectionFactory
     * @param Stock $stock
     * @param StockRegistryInterface $stockRegistry
     */
    public function __construct(
        LoggerInterface        $logger,
        ConfigurationHelper    $configurationHelper,
        CollectionFactory      $productCollectionFactory,
        Stock                  $stock,
        StockRegistryInterface $stockRegistry
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->stockFilter = $stock;
        $this->stockRegistry = $stockRegistry;
    }

    /**
     * @param $sku
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function process($sku = null)
    {
        if ($this->configuration->useCustomStock() && $this->configuration->getCustomStockId()) {
            // caso custom inventory
            $this->processCustomInventory($sku);

            return;
        }

        // caso inventory standard
        $this->processDefaultInventory($sku);
    }

    /**
     * @param $sku
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function processCustomInventory($sku)
    {
        $collection = $this->productCollectionFactory->create()
            ->addAttributeToSelect('*')
            ->joinTable(
                ['stock' => 'inventory_stock_' . $this->configuration->getCustomStockId()],
                'sku=sku',
                ['quantity as qty']
            )
            ->addAttributeToFilter('status', ProductStatus::STATUS_ENABLED)
            ->addAttributeToFilter('type_id', ConfigurableProduct::TYPE_CODE);
        if ($this->configuration->isConfigurableResetStockOnlyOutOfStock()) {
            $collection->addFieldToFilter('is_saleable', ['eq' => 0]);
        }
        if ($sku) {
            $collection->addFieldToFilter('sku', ['eq' => $sku]);
        }

        foreach ($collection as $product) {
            $haveAvailableChild = $product->getQty() > 0 ? true : false;
            $this->updateInStockStatus($product, $haveAvailableChild);
        }
    }

    /**
     * @param $sku
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function processDefaultInventory($sku)
    {
        // prendo tutti i configurabili con stato outofstock, abilitati e prezzo>0
        $collection = $this->productCollectionFactory->create()
            ->addAttributeToSelect('*')
            ->joinField('qty',
                'cataloginventory_stock_item',
                'qty',
                'product_id=entity_id',
                '{{table}}.stock_id=1',
                'left'
            )
            //->joinTable('cataloginventory_stock_item', 'product_id=entity_id', ['stock_status' => 'is_in_stock'])
            ->addAttributeToSelect('stock_status')
            ->addFieldToFilter('type_id', ['eq' => ConfigurableProduct::TYPE_CODE])
            ->addAttributeToFilter('status', ['eq' => Status::STATUS_ENABLED]);
        if ($this->configuration->isConfigurableResetStockOnlyOutOfStock()) {
            $collection->addFieldToFilter('stock_status', ['eq' => 0]);
        }
        if ($sku) {
            $collection->addFieldToFilter('sku', ['eq' => $sku]);
        }

        foreach ($collection as $product) {
            $children = $product->getTypeInstance()->getUsedProducts($product);
            $this->updateInStockStatus($product, $this->haveAvailableChilds($children));
        }
    }

    /**
     * @param $childs
     * @return bool
     */
    protected function haveAvailableChilds($childs): bool
    {
        foreach ($childs as $child) {
            $productStock = $this->stockRegistry->getStockItem($child->getId());

            if ($child->getStatus() == Status::STATUS_ENABLED && $child->getPrice() > 1 && $productStock->getQty() > 0) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param $product
     * @param $haveAvailableChilds
     */
    public function updateInStockStatus($product, $haveAvailableChilds)
    {
        try {
//            $stockItem = $this->stockRegistry->getStockItemBySku($product->getSku());
//            $stockItem->setIsInStock($haveAvailableChilds);
//            $this->stockRegistry->updateStockItemBySku($product->getSku(), $stockItem);
            if ($stockItem = $this->stockRegistry->getStockItemBySku($product->getSku())) {
                $stockItem->setIsInStock($haveAvailableChilds);
                $stockItem->setManageStock(1);
                $stockItem->setStockStatusChangedAuto(1);
                $stockItem->save();
            }
            if ($stockStatus = $this->stockRegistry->getStockStatusBySku($product->getSku())) {
                $stockStatus->setStockStatus(1);
                $stockStatus->save();
            }
        } catch (\Exception $e) {
            $this->logger->error("Error while updating stock product - {$e->getMessage()}");
        }
    }
}
