<?php

namespace Drop\NotifyBankTransfer\Observer\Order;

class CancelAfter implements \Magento\Framework\Event\ObserverInterface
{
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Drop\NotifyBankTransfer\Helper\Email
     */
    protected $email;

    /**
     * @var \Drop\NotifyBankTransfer\Logger\Logger
     */
    protected $logger;

    /**
     * @var \Magento\Framework\Url $urlHelpe
     */
    protected $url;

    /**
     * Construct
     * 
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig 
     * @param \Drop\NotifyBankTransfer\Logger\Logger            $logger      
     * @param \Drop\NotifyBankTransfer\Helper\Email             $email       
     * @param \Magento\Framework\Url                             $url         
     */
    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Drop\NotifyBankTransfer\Logger\Logger            $logger,
        \Drop\NotifyBankTransfer\Helper\Email             $email,
        \Magento\Framework\Url                             $url
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->logger      = $logger;
        $this->email       = $email;
        $this->url         = $url;
    }

    /**
     * {@inheritDoc}
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $order         = $observer->getOrder();
        $incrementId   = $observer->getOrder()->getIncrementId();
        $orderId       = $observer->getOrder()->getEntityId();
        $paymentMethod = $observer->getOrder()->getPayment()->getMethod();
        $email         = $observer->getOrder()->getShippingAddress()->getEmail();

        if (strtolower($paymentMethod) != \Magento\OfflinePayments\Model\Banktransfer::PAYMENT_METHOD_BANKTRANSFER_CODE) {
            return $this;
        }

        try {
            $this->email->sendNotify(
                'template_drop_notify_bank_transfer_order_cancel',
                array(
                    'order'        => $order,
                    'customerName' => strtolower($order->getCustomerName()) == "guest" ? $order->getShippingAddress()->getFirstname() . " " . $order->getShippingAddress()->getLastname() : $order->getCustomerName(),
                    'viewOrderUrl' => $this->url->getUrl(
                        'sales/order/view', [
                            'order_id' => $orderId,
                            '_nosid'   => true,
                        ]
                    ),
                    'storeName'    => $this->scopeConfig->getValue(
                        'trans_email/ident_sales/name',
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                    ),
                    'store_email'  => $this->scopeConfig->getValue(
                        'trans_email/ident_sales/email',
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                    ),
                    'store_phone'  => $this->scopeConfig->getValue(
                        'general/store_information/phone',
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                    ),
                    'store_hours'  => $this->scopeConfig->getValue(
                        'general/store_information/hours',
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                    )
                ),
                $email
            );
        } catch (\Exception $e) {
            $this->logger->critical(
                "Mancato invio email order_cancel a " . $email . " per ordine " . $incrementId . " - " . __METHOD__
            );

            $this->logger->critical($e);
            throw $e;
        }

        $this->logger->info(
            "Inviata email order_cancel a " . $email . " per ordine " . $incrementId . " - " . __METHOD__
        );
    }
}
