<?php

namespace Drop\NotifyBankTransfer\Model;

class RunCheckPendingTransferMoney extends \Magento\Framework\Model\AbstractModel
{
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resource;

    /**
     * @var \Drop\NotifyBankTransfer\Model\NotifyBankTransferFactory
     */
    protected $notifyBankTransferFactory;

    /**
     * @var \Drop\NotifyBankTransfer\Model\Repository\OrderRepository
     */
    protected $orderRepository;

    /**
     * @var \Drop\NotifyBankTransfer\Logger\Logger
     */
    protected $logger;

    /**
     * @var \Drop\NotifyBankTransfer\Helper\Time
     */
    protected $time;

    /**
     * @var \Drop\NotifyBankTransfer\Helper\Email
     */
    protected $email;

    /**
     * @var \Drop\NotifyBankTransfer\Helper\Data
     */
    protected $data;

    /**
     * @var \Drop\NotifyBankTransfer\Helper\Database
     */
    protected $database;

    /**
     * Construct
     *
     * @param \Magento\Framework\App\Config\ScopeConfigInterface        $scopeConfig
     * @param \Magento\Framework\App\ResourceConnection                 $resource
     * @param \Drop\NotifyBankTransfer\Model\NotifyBankTransferFactory  $notifyBankTransferFactory
     * @param \Drop\NotifyBankTransfer\Model\Repository\OrderRepository $orderRepository
     * @param \Drop\NotifyBankTransfer\Logger\Logger                    $logger
     * @param \Drop\NotifyBankTransfer\Helper\Time                      $time
     * @param \Drop\NotifyBankTransfer\Helper\Email                     $email
     * @param \Drop\NotifyBankTransfer\Helper\Data                      $data
     * @param \Drop\NotifyBankTransfer\Helper\Database                  $database
     */
    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface        $scopeConfig,
        \Magento\Framework\App\ResourceConnection                 $resource,
        \Drop\NotifyBankTransfer\Model\NotifyBankTransferFactory  $notifyBankTransferFactory,
        \Drop\NotifyBankTransfer\Model\Repository\OrderRepository $orderRepository,
        \Drop\NotifyBankTransfer\Logger\Logger                    $logger,
        \Drop\NotifyBankTransfer\Helper\Time                      $time,
        \Drop\NotifyBankTransfer\Helper\Email                     $email,
        \Drop\NotifyBankTransfer\Helper\Data                      $data,
        \Drop\NotifyBankTransfer\Helper\Database                  $database
    ) {
        $this->scopeConfig               = $scopeConfig;
        $this->resource                  = $resource;
        $this->notifyBankTransferFactory = $notifyBankTransferFactory;
        $this->orderRepository           = $orderRepository;
        $this->logger                    = $logger;
        $this->time                      = $time;
        $this->email                     = $email;
        $this->data                      = $data;
        $this->database                  = $database;
    }

    /**
     * Run check
     *
     * @return void
     */
    public function run()
    {
        $collection = $this->orderRepository->getOrdersWithPendingTransferMoney();

        if ( ! $collection->getSize()) {
            return;
        }

        $connection = $this->resource->getConnection();

        $connection->beginTransaction();

        foreach ($collection as $order) {
            try {
                $incrementId = $order->getIncrementId();
                $email       = $order->getShippingAddress()->getEmail();

                $notify = $this->notifyBankTransferFactory->create()->load(
                    $incrementId,
                    'increment_id'
                );

                $payloadCurrentTimestamp = json_decode(
                    $notify->getNextTimestampSendPayload(),
                    true
                );

                $this->email->sendNotify(
                    $this->email->getTemplateEmailByDay(
                        $payloadCurrentTimestamp['day']
                    ),
                    array(
                        'infoBonifico' => $order->getPayment()->getAdditionalInformation()["instructions"],
                        'days'         => (string) $payloadCurrentTimestamp['day'],
                        'customerName' => strtolower($order->getCustomerName()) == "guest" ? $order->getShippingAddress()->getFirstname() . " " . $order->getShippingAddress()->getLastname() : $order->getCustomerName(),
                        'order'        => $order,
                        'storeName'    => $this->scopeConfig->getValue(
                            'trans_email/ident_sales/name',
                            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                        ),
                        'store_email'  => $this->scopeConfig->getValue(
                            'trans_email/ident_sales/email',
                            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                        ),
                        'store_phone'  => $this->scopeConfig->getValue(
                            'general/store_information/phone',
                            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                        ),
                        'store_hours'  => $this->scopeConfig->getValue(
                            'general/store_information/hours',
                            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                        ),
                    ),
                    $email
                );

                list($timestamp, $timestampDay, $lastTimestampPayload, $queue, $lastNotify) = $this->database->elaborateQueue(
                    $notify->getQueueTimestampSend(),
                    $notify->getNextTimestampSendPayload()
                );

                $notify->setNextTimestampSend(
                    $timestamp
                );

                $notify->setNextTimestampSendPayload(
                    $lastTimestampPayload
                );

                $notify->setQueueTimestampSend(
                    $queue
                );

                $notify->setIsSentLastNotify(
                    $lastNotify
                );

                $notify->save();
            } catch (\Exception $e) {
                $connection->rollBack();
                $this->logger->critical($e);
                throw $e;
            }

            $this->logger->info(
                "Eseguo logica aggiornamento queue " . json_encode([$timestamp, $timestampDay, $this->email->getTemplateEmailByDay($timestampDay), $lastTimestampPayload, $queue, $lastNotify]) . " - " . __METHOD__
            );

            $this->logger->info(
                "Inviata email reminder (templateId: " . $this->email->getTemplateEmailByDay($payloadCurrentTimestamp['day']) . ") a " . $email . " per ordine " . $incrementId . " - " . __METHOD__
            );
        }

        $connection->commit();
    }
}
