<?php

namespace Drop\MirasvitEmail\Model\Queue;


use Magento\Framework\App\Area;
use Magento\Framework\App\ProductMetadataInterface as ProductMetadata;
use Magento\Framework\App\State as AppState;
use Magento\Framework\Mail\MessageFactory as MailMessageFactory;
use Magento\Framework\Mail\MessageInterface;
use Magento\Framework\Mail\TransportInterfaceFactory;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Store\Model\App\Emulation;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Module\Manager;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Service\Queue\MailModifierInterface;
use Mirasvit\Email\Controller\RegistryConstants;
use Mirasvit\Email\Helper\Data as Helper;
use Mirasvit\Email\Model\Config;
use Mirasvit\Email\Model\Queue;
use Mirasvit\Email\Model\ResourceModel\Queue\CollectionFactory as QueueCollectionFactory;
use Mirasvit\Email\Model\Unsubscription;

/**
 * Sender class
 */
class Sender extends \Mirasvit\Email\Model\Queue\Sender
{
    /**
     * @var Unsubscription
     */
    protected $unsubscription;

    /**
     * @var QueueCollectionFactory
     */
    protected $queueCollectionFactory;

    /**
     * @var Config
     */
    protected $config;

    /**
     * @var MailMessageFactory
     */
    protected $mailMessageFactory;

    /**
     * @var Emulation
     */
    protected $appEmulation;

    /**
     * @var AppState
     */
    protected $appState;

    /**
     * @var \Magento\Framework\Mail\TransportInterfaceFactory
     */
    protected $mailTransportFactory;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Manager
     */
    protected $moduleManager;

    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var \Magento\Framework\Mail\Template\TransportBuilder
     */
    private $transportBuilder;

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var  MailModifierInterface[]
     */
    private $modifiers;

    /**
     * @var ProductMetadata
     */
    private $productMetadata;
    /**
     * @var DateTime
     */
    private $date;

    /**
     * Sender constructor.
     * @param Registry $registry
     * @param TransportBuilder $transportBuilder
     * @param Unsubscription $unsubscription
     * @param QueueCollectionFactory $queueCollectionFactory
     * @param Config $config
     * @param DateTime $date
     * @param MailMessageFactory $mailMessageFactory
     * @param Emulation $appEmulation
     * @param AppState $appState
     * @param TransportInterfaceFactory $mailTransportFactory
     * @param StoreManagerInterface $storeManager
     * @param Manager $moduleManager
     * @param Helper $helper
     * @param ProductMetadata $productMetadata
     * @param array $modifiers
     */
    public function __construct(
        Registry                  $registry,
        \Mirasvit\Email\Model\Queue\TransportBuilder          $transportBuilder,
        Unsubscription            $unsubscription,
        QueueCollectionFactory    $queueCollectionFactory,
        Config                    $config,
        DateTime                  $date,
        MailMessageFactory        $mailMessageFactory,
        Emulation                 $appEmulation,
        AppState                  $appState,
        TransportInterfaceFactory $mailTransportFactory,
        StoreManagerInterface     $storeManager,
        Manager                   $moduleManager,
        Helper                    $helper,
        ProductMetadata           $productMetadata,
        array                     $modifiers = []
    ) {
        $this->registry               = $registry;
        $this->unsubscription         = $unsubscription;
        $this->queueCollectionFactory = $queueCollectionFactory;
        $this->config                 = $config;
        $this->date                   = $date;
        $this->mailMessageFactory     = $mailMessageFactory;

        $this->appEmulation         = $appEmulation;
        $this->appState             = $appState;
        $this->mailTransportFactory = $mailTransportFactory;
        $this->storeManager         = $storeManager;
        $this->moduleManager        = $moduleManager;
        $this->helper               = $helper;
        $this->transportBuilder     = $transportBuilder;
        $this->modifiers            = $modifiers;
        $this->productMetadata      = $productMetadata;

        parent::__construct(
            $registry,
            $transportBuilder,
            $unsubscription,
            $queueCollectionFactory,
            $config,
            $date,
            $mailMessageFactory,
            $appEmulation,
            $appState,
            $mailTransportFactory,
            $storeManager,
            $moduleManager,
            $helper,
            $productMetadata,
            $modifiers);
    }

    /**
     * @return \Drop\Email\Model\TransportBuilder
     */
    private function getTransportBuilder()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
            ->get(\Magento\Framework\Mail\Template\TransportBuilder::class);
    }


    /**
     * {@inheritDoc}
     */
    public function send(
        $queue,
        $force = false
    ) {
        if ( ! $this->canSend($queue) && ! $force) {
            return false;
        }

        $dropTransportBuilder = $this->getTransportBuilder();

        // register current email queue model instance
        $this->registry->register(\Mirasvit\Email\Controller\RegistryConstants::CURRENT_QUEUE, $queue, true);

        $this->appEmulation->startEnvironmentEmulation($queue->getArgs('store_id'), \Magento\Framework\App\Area::AREA_FRONTEND, true);
        $subject = $queue->getMailSubject();
        $this->appEmulation->stopEnvironmentEmulation();

        $this->appEmulation->startEnvironmentEmulation($queue->getArgs('store_id'), \Magento\Framework\App\Area::AREA_FRONTEND, true);

        $body = $queue->getMailContent();

        foreach ($this->modifiers as $modifier) {
            $body = $modifier->modifyContent($queue, $body);
        }

        $body = $this->helper->prepareQueueContent($body, $queue);
        $this->appEmulation->stopEnvironmentEmulation();

        $this->appEmulation->startEnvironmentEmulation($queue->getArgs('store_id'), \Magento\Framework\App\Area::AREA_FRONTEND, true);

        $recipients = explode(',', $queue->getRecipientEmail());
        if ($this->config->isSandbox() && ! $queue->getArg('force')) {
            $recipients = explode(',', $this->config->getSandboxEmail());
        }

        foreach ($recipients as $index => $email) {
            $name = $queue->getRecipientName();
            if (count($recipients) > 1) {
                $name .= ' - ' . ($index + 1);
            }
            unset($recipients[$index]);
            $recipients[$name] = $email;
        }

        //trim spaces and remove all empty items
        $copyTo = array_filter(array_map('trim', explode(',', $queue->getTrigger()->getCopyEmail())));
        foreach ($copyTo as $bcc) {
            $dropTransportBuilder->addBcc($bcc);
        }

        $dropTransportBuilder
            ->setReplyTo($queue->getSenderEmail(), $queue->getSenderName());

        $magentoVersion = $this->productMetadata->getVersion();
        if (version_compare($magentoVersion, '2.2.8', '>=')) {
            $dropTransportBuilder->setFrom([
                'name'  => $queue->getSenderName(),
                'email' => $queue->getSenderEmail(),
            ]);
        } else {
            $dropTransportBuilder->setFrom($queue->getSenderEmail(), $queue->getSenderName());
        }

        foreach ($recipients as $name => $email) {
            $dropTransportBuilder->addTo($email, $name);
        }

        $dropTransportBuilder->setTemplateIdentifier(
            'raw_template_mirasvit'
        );

        $dropTransportBuilder->setTemplateVars([
            'data'    => $body,
            'subject' => $subject,
        ]);

        $dropTransportBuilder->setTemplateOptions([
            'area'  => \Magento\Framework\App\Area::AREA_FRONTEND,
            'store' => $queue->getArgs('store_id'),
        ]);

        $transport = $dropTransportBuilder->getTransport();

        $transport->sendMessage();

        $queue->delivery();

        $this->appEmulation->stopEnvironmentEmulation();

        return true;
    }
}
