<?php

namespace Drop\Klarna\Observer;

use Drop\Klarna\Api\PaymentInterface;
use Drop\Klarna\Helper\Data;
use Magento\Framework\DB\TransactionFactory;
use Magento\Framework\Event\Observer;
use Magento\Payment\Observer\AbstractDataAssignObserver;
use Magento\Sales\Model\Order\Invoice;

/**
 * Class DataAssignObserver
 */
class CheckoutSubmitAllAfterObserver extends AbstractDataAssignObserver implements PaymentInterface {

    /**
     * @var Data
     */
    private $klarnaHelper;
    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * AfterOrderObserver constructor.
     *
     * @param Data $klarnaHelper
     * @param TransactionFactory $transactionFactory
     */
    public function __construct(
        Data $klarnaHelper,
        TransactionFactory $transactionFactory
    ) {
        $this->klarnaHelper = $klarnaHelper;
        $this->transactionFactory = $transactionFactory;
    }

    /**
     * @param Observer $observer
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(Observer $observer) {
        /**
         * @var $order \Magento\Sales\Model\Order
         */
        $order = $observer->getEvent()->getOrder();

        if (!$this->isPaymentMethodKlarna($order)){
            return $this;
        }

        $captureMode = $this->klarnaHelper->getCaptureMode();
        if ($captureMode != self::CAPTURE_MODE_ORDER) {
            return $this;
        }

        //Nel caso venga aggiornato lo status per sospetta frode dopo aver impostato Capture on Order
        if ($order->getStatus() == \Magento\Sales\Model\Order::STATUS_FRAUD) {
            return $this;
        }

        $this->klarnaHelper->logDebug("START - CheckoutSubmitAllAfter Observer");

        if (!$order->canInvoice()) {
            $this->klarnaHelper->log("--- CheckoutSubmitAllAfter Observer");
            $this->klarnaHelper->log("--- Error, can't create invoice for order " . $order->getId());
            throw new \Exception("Error, can't create invoice for order " . $order->getId());
        }

        $captureError = false;
        try {
            $this->klarnaHelper->logDebug("--- Prepare Invoice - CAPTURE ONLINE - order " . $order->getId());
            $invoice = $order->prepareInvoice();
            $invoice->getOrder()->setIsInProcess(true);
            $invoice->setRequestedCaptureCase(Invoice::CAPTURE_ONLINE);
            $this->klarnaHelper->logDebug("--- Invoice Register & Pay");
            $invoice->register()->pay();
            $this->klarnaHelper->logDebug("--- Invoice Save");
            $invoice->save();
            $this->klarnaHelper->logDebug("--- Transaction Factory Create & addObjects");
            $transactionSave = $this->transactionFactory->create()->addObject($invoice)->addObject($invoice->getOrder());
            $this->klarnaHelper->logDebug("--- Transaction Save");
            $transactionSave->save();
            $this->klarnaHelper->logDebug("END - SalesOrderShipmentSaveBefore Observer");
            return $this;
        } catch (\Exception $e) {
            $this->klarnaHelper->log("--- CheckoutSubmitAllAfter Observer");
            $this->klarnaHelper->log("--- Exception:" . $e->getMessage());
            if(strpos($e->getMessage(), "remaining authorized amount is 0") !== false){
                $captureError = true;
            }
        }

        try {
            if ($captureError) {
                $this->klarnaHelper->logDebug("--- Prepare Invoice - CAPTURE OFFLINE - order " . $order->getId());
                $invoice = $order->prepareInvoice();
                $invoice->getOrder()->setIsInProcess(true);
                $invoice->setRequestedCaptureCase(Invoice::CAPTURE_OFFLINE);
                $this->klarnaHelper->logDebug("--- Invoice Register & Pay");
                $invoice->register()->pay();
                $this->klarnaHelper->logDebug("--- Invoice Save");
                $invoice->save();
                $this->klarnaHelper->logDebug("--- Transaction Factory Create & addObjects");
                $transactionSave = $this->transactionFactory->create()->addObject($invoice)->addObject($invoice->getOrder());
                $this->klarnaHelper->logDebug("--- Transaction Save");
                $transactionSave->save();
            }
        } catch (\Exception $e) {
            $this->klarnaHelper->log("--- CheckoutSubmitAllAfter Observer");
            $this->klarnaHelper->log("--- Exception:" . $e->getMessage());
            throw new \Exception('Error saving invoice. The error message is:' . $e->getMessage());
        }

        $this->klarnaHelper->logDebug("END - CheckoutSubmitAllAfter Observer");
        return $this;
    }

    /**
     * Determine if the payment method is Klarna
     */
    public function isPaymentMethodKlarna($order) {
        return strpos($order->getPayment()->getMethod(), 'klarna') !== false;
    }
}
