<?php

namespace Drop\Klarna\Observer;

use Drop\Klarna\Api\PaymentInterface;
use Drop\Klarna\Helper\Data;
use Magento\Framework\DB\TransactionFactory;
use Magento\Framework\Event\Observer;
use Magento\Payment\Observer\AbstractDataAssignObserver;
use Magento\Sales\Model\Order\Invoice;

/**
 * Class DataAssignObserver
 */
class SalesOrderShipmentSaveBeforeObserver extends AbstractDataAssignObserver implements PaymentInterface {

    /**
     * @var Data
     */
    private $klarnaHelper;
    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * BeforeShipmentObserver constructor.
     *
     * @param Data $klarnaHelper
     * @param TransactionFactory $transactionFactory
     */
    public function __construct(
        Data $klarnaHelper,
        TransactionFactory $transactionFactory
    ) {
        $this->klarnaHelper = $klarnaHelper;
        $this->transactionFactory = $transactionFactory;
    }

    /**
     * @param Observer $observer
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(Observer $observer) {
        $shipment = $observer->getEvent()->getShipment();
        $order = $shipment->getOrder();

        if (!$this->isPaymentMethodKlarna($order)) {
            return $this;
        }

        $captureMode = $this->klarnaHelper->getCaptureMode();
        if ($captureMode != self::CAPTURE_MODE_SHIPMENT) {
            return $this;
        }

        if (!$order->canInvoice()) {
            throw new \Exception(sprintf("Error, can't create invoice."));
        }

        $payment = $order->getPayment();
        try {
            $invoice = $order->prepareInvoice();
            $invoice->getOrder()->setIsInProcess(true);
            $invoice->setRequestedCaptureCase(Invoice::CAPTURE_ONLINE);
            $invoice->register()->pay();
            $invoice->save();
            $transactionSave = $this->transactionFactory->create()->addObject($invoice)->addObject($invoice->getOrder());
            $transactionSave->save();
        } catch (\Exception $e) {
            throw new \Exception(sprintf('Error saving invoice. The error message is:', $e->getMessage()));
        }
        return $this;
    }

    /**
     * Determine if the payment method is Klarna
     */
    public function isPaymentMethodKlarna($order) {
        return strpos($order->getPayment()->getMethod(), 'klarna') !== false;
    }
}
