<?php
declare(strict_types=1);

namespace Drop\IubendaConsentSolution\Observer\Newsletter\Subscriber;

use Drop\IubendaConsentSolution\Helper\Data as HelperData;
use Drop\IubendaConsentSolution\Model\Api\Client as ApiClient;
use Magento\Customer\Model\Session;
use Magento\Framework\App\RequestInterface;

class SaveAfter implements \Magento\Framework\Event\ObserverInterface {
    /**
     * @var ApiClient
     */
    protected $apiClient;

    /**
     * @var HelperData
     */
    protected $helperData;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * @var Session
     */
    protected $customerSession;

    /**
     * @param ApiClient $apiClient
     * @param HelperData $helperData
     * @param RequestInterface $request
     * @param Session $customerSession
     */
    public function __construct(
        ApiClient $apiClient,
        HelperData $helperData,
        RequestInterface $request,
        Session $customerSession
    ) {
        $this->apiClient = $apiClient;
        $this->helperData = $helperData;
        $this->request = $request;
        $this->customerSession = $customerSession;
    }

    /**
     * Execute observer
     *
     * @param \Magento\Framework\Event\Observer $observer
     * @return void
     */
    public function execute(
        \Magento\Framework\Event\Observer $observer
    ) {
        if(!$this->helperData->isModuleEnabled()){
            return $this;
        }

        $this->helperData->logDebug("START - Newsletter Subscriber SaveAfter Observer");

        try {
            //Session flag per impedire l'esecuzione dell'observer più volte
            if($this->customerSession->getIubendaSubscriberSaveAfterObserverExecuted()){
                //Reset flag per successivi salvataggi
                $this->customerSession->setIubendaSubscriberSaveAfterObserverExecuted(false);
                //this method has already been executed once in this request (see comment below)
                $this->helperData->logDebug("--- getIubendaSubscriberSaveAfterObserverExecuted = true");
                $this->helperData->logDebug("END - Subscriber SaveAfter Observer");
                return $this;
            }

            $subscriber = $observer->getSubscriber();
            //esce se non sono state fatte modifiche
            if(!$subscriber->hasDataChanges() || !$subscriber->isStatusChanged()) {
                $this->helperData->logDebug("--- No data changes");
                $this->helperData->logDebug("END - Newsletter Subscriber SaveAfter Observer");
                return $this;
            }

            $subscriberEmail = $subscriber->getEmail();
            //escludo i subscriber senza email, sono dovuti a elaborazioni da backoffice o importazioni errate,
            //non ha senso registrare un consenso se non si ha un indirizzo email.
            if(empty($subscriberEmail)){
                $this->helperData->logDebug("--- Empty subscriber email");
                $this->helperData->logDebug("END - Newsletter Subscriber SaveAfter Observer");
                return $this;
            }

            //lista di tutti i subject con la stessa email, il valore viene aggiornato per tutti
            $list = $this->apiClient->listSubjectsByEmail($subscriberEmail);
            $this->helperData->logDebug("--- subject list: " . print_r($list, true));
            foreach ($list as $subject) {
                //aggiorno i campi di tutti i subject in modo che siano allineati
                $this->apiClient->updateSubjectFromSubscriber($subscriber, $subject["id"]);
                //crea o aggiorna i consensi nell'archivio Iubenda Consents
                $this->apiClient->createNewsletterConsent($subscriber, $subject["id"]);
            }

            //Customer Addresses seems to call the before_save event twice,
            //so we need to set a variable so we only process it once, otherwise we get duplicates
            $this->customerSession->setIubendaSubscriberSaveAfterObserverExecuted(true);
        } catch (Exception $ex) {
            $this->helperData->log("--- Newsletter Subscriber SaveAfter observer");
            $this->helperData->log("--- error: " . $ex->getMessage());
        }

        $this->helperData->logDebug("END - Newsletter Subscriber SaveAfter Observer");
        return $this;
    }
}

