<?php
declare(strict_types=1);
namespace Drop\IubendaConsentSolution\Model\Api;

use Drop\IubendaConsentSolution\Helper\Data as HelperData;
use Magento\Store\Model\StoreManagerInterface as StoreManager;
use Magento\Framework\HTTP\Client\Curl as Curl;

class Client {
    /**
     * API Endpoints
     */
    public const CONSENT_ENDPOINT = "consent";
    public const SUBJECTS_ENDPOINT = "subjects";

    /**
     * @var HelperData
     */
    protected $helperData;
    /**
     * @var StoreManager
     */
    protected $_storeManager;
    /**
     * @var Curl
     */
    protected $curl;

    /**
     * Consentsolution constructor.
     *
     * @param HelperData $helperData
     * @param StoreManager $storeManager
     * @param Curl $curl
     */
    public function __construct(
        HelperData $helperData,
        StoreManager $storeManager,
        Curl $curl
    ) {
        $this->helperData = $helperData;
        $this->_storeManager = $storeManager;
        $this->curl = $curl;
    }

    /**
     * Get store identifier
     *
     * @return  int
     */
    public function getStoreId() {
        return $this->_storeManager->getStore()->getId();
    }

    /**
     * Private Api Key
     *
     * @return bool|mixed
     */
    public function getPrivateApiKey() {
        return $this->helperData->getConfigValue($this->helperData::PRIVATE_API_KEY_CONFIG_PATH);
    }

    /**
     * Set Options in Curl Object
     */
    public function setCurlOptions() {
        //reset options
        $this->curl->setOptions([]);
        $this->curl->setOption(CURLOPT_RETURNTRANSFER, true);
    }

    /**
     * Add Headers to Curl Object
     */
    public function addCurlHeaders() {
        $this->curl->addHeader("ApiKey", $this->getPrivateApiKey());
        $this->curl->addHeader("Content-Type", "application/json");
    }

    /**
     * Http Api Endpoint
     *
     * @return bool|mixed
     */
    public function getHttpApiEndpoint() {
        return $this->helperData->getConfigValue($this->helperData::HTTP_API_ENDPOINT_CONFIG_PATH);
    }

    /**
     * Newsletter Input Name
     *
     * @return mixed
     */
    public function getNewsletterInputName() {
        return $this->helperData->getConfigValue($this->helperData::NEWSLETTER_INPUT_NAME_CONFIG_PATH);
    }

    /**
     * Privacy Input Name
     *
     * @return mixed
     */
    public function getProfilazioneInputName() {
        return $this->helperData->getConfigValue($this->helperData::PROFILAZIONE_INPUT_NAME_CONFIG_PATH);
    }

    /**
     * Prepare data array from customer
     *
     * @param \Magento\Customer\Model\Customer $customer
     * @return array
     */
    public function prepareCustomerData(\Magento\Customer\Model\Customer $customer) {
        try {
            $this->helperData->logDebug("------ Preparing data for customer " . $customer->getId());

            //assistance_allowed può avere valore 1 o 2, rispettivamente false e true - solo Mage 2.4+
            $assistanceAllowed = "false";
            if($customer->getAssistanceAllowed()) {
                $assistanceAllowed = var_export(boolval($customer->getAssistanceAllowed() - 1), true);
            }

            //newsletter subscribed non è nel customer in alcune versione di Magento, fallback su REQUEST
            $isSubscribed = "false";
            $newsletterInputName = $this->getNewsletterInputName();
            if(!empty($_REQUEST[$newsletterInputName])) {
                $isSubscribed = var_export(boolval($_REQUEST[$newsletterInputName]), true);
            }
            if($customer->getIsSubscribed()) {
                $isSubscribed = var_export($customer->getIsSubscribed(), true);
            }

            //profilazione può avere valore 0 o 1, rispettivamente false e true
            $profilazione = "false";
            $profilazioneInputName = $this->getProfilazioneInputName();
            if($customer->getData($profilazioneInputName)) {
                $profilazione = var_export(boolval($customer->getData($profilazioneInputName)), true);
            }

            //privacy è accettata di default
            $privacy = var_export(true, true);

            $data = [
                "subject" => array(
                    "id" => $customer->getId(),
                    "email" => $customer->getEmail(),
                    "first_name" => $customer->getFirstname(),
                    "last_name" => $customer->getLastname(),
                    "full_name" => $customer->getLastname() . " " . $customer->getFirstname(),
                    "verified" => "false"
                ),
                "legal_notices" => array(
                    array(
                        "identifier" => "assistance_allowed"
                    ),
                    array(
                        "identifier" => "newsletter"
                    ),
                    array(
                        "identifier" => "privacy_policy"
                    ),
                    array(
                        "identifier" => "profilazione"
                    )
                ),
                "proofs" => array(
                    array(
                        "content" => "checkbox",
                        "form" => "form"
                    )
                ),
                "preferences" => array(
                    "assistance_allowed" => $assistanceAllowed,
                    "newsletter" => $isSubscribed,
                    "privacy_policy" => $privacy,
                    "profilazione" => $profilazione
                ),
                "ip_address" => $this->helperData->getRemoteAddress(),
                "autodetect_ip_address" => "false"
            ];

            $this->helperData->logDebug("------ Data:" . print_r($data, true));
        } catch (Exception $ex) {
            $this->helperData->log("------ prepareCustomerData method");
            $this->helperData->log("------" . $ex->getMessage());
            $this->helperData->log("------ Data:" . print_r($data, true));
        }
        return $data;
    }

    /**
     * Create subject and its relative preferences
     *
     * @param Magento\Customer\Model\Customer $customer
     * @return mixed
     */
    public function createConsent(\Magento\Customer\Model\Customer $customer) {
        $this->helperData->logDebug("--- START - Client createConsent");

        try {
            //setup
            $this->setCurlOptions();
            $this->addCurlHeaders();

            //params
            $data = $this->prepareCustomerData($customer);
            $url = $this->getHttpApiEndpoint() . $this::CONSENT_ENDPOINT;
            $this->curl->post($url, json_encode($data));

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ createConsent method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ data: " . print_r($data, true));
        }
        $this->helperData->logDebug("--- END - Client createConsent");
        return $response;
    }

    /**
     * Prepare data array from subscriber
     *
     * @param \Magento\Newsletter\Model\Subscriber $subscriber
     * @param $subjectId
     * @return array
     */
    public function prepareSubscriberData(\Magento\Newsletter\Model\Subscriber $subscriber, $subjectId) {
        try {
            $this->helperData->logDebug("------ Preparing data for subscriber " . $subscriber->getId());

            //fallback su REQUEST per coprire alcune form
            $isSubscribed = "false";
            $newsletterInputName = $this->getNewsletterInputName();
            if(!empty($_REQUEST[$newsletterInputName])) {
                $isSubscribed = var_export(boolval($_REQUEST[$newsletterInputName]), true);
            }
            if($subscriber->getSubscriberStatus() == \Magento\Newsletter\Model\Subscriber::STATUS_SUBSCRIBED) {
                $isSubscribed = "true";
            }

            $data = [
                "subject" => array(
                    "id" => $subjectId,
                    "email" => $subscriber->getEmail()
                ),
                "legal_notices" => array(
                    array(
                        "identifier" => "newsletter"
                    )
                ),
                "proofs" => array(
                    array(
                        "content" => "checkbox",
                        "form" => "form"
                    )
                ),
                "preferences" => array(
                    "newsletter" => $isSubscribed
                ),
                "ip_address" => $this->helperData->getRemoteAddress(),
                "autodetect_ip_address" => "false"
            ];
            $this->helperData->logDebug("------ Data:" . print_r($data, true));
        } catch (Exception $ex) {
            $this->helperData->log("------ prepareSubscriberData method");
            $this->helperData->log("------" . $ex->getMessage());
            $this->helperData->log("------ Data:" . print_r($data, true));
        }
        return $data;
    }

    /**
     * Create subject and its relative newsletter preferences
     *
     * @param Magento\Newsletter\Model\Subscriber $subscriber
     * @param $subjectId
     * @return mixed
     */
    public function createNewsletterConsent(\Magento\Newsletter\Model\Subscriber $subscriber, $subjectId) {
        $this->helperData->logDebug("--- START - Client createNewsletterConsent");

        try {
            //setup
            $this->setCurlOptions();
            $this->addCurlHeaders();

            //params
            $data = $this->prepareSubscriberData($subscriber, $subjectId);
            $url = $this->getHttpApiEndpoint() . $this::CONSENT_ENDPOINT;
            $this->curl->post($url, json_encode($data));

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ createNewsletterConsent method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ data: " . print_r($data, true));
        }
        $this->helperData->logDebug("--- END - Client createNewsletterConsent");
        return $response;
    }

    /**
     * Search subject data from Iubenda Consent Solution filtering by eMail
     *
     * @param $email
     * @return mixed|null
     */
    public function listSubjectsByEmail($email) {
        $this->helperData->logDebug("--- START - Client listSubjects");

        try {
            //setup
            $this->setCurlOptions();
            $this->addCurlHeaders();

            //params
            $data = [
                "email_exact" => $email
            ];
            $url = $this->getHttpApiEndpoint() . $this::SUBJECTS_ENDPOINT . "?" . http_build_query($data);
            $this->curl->get($url);

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ listSubjects method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ email: " . $email);
        }
        $this->helperData->logDebug("--- END - Client listSubjects");
        return $response;
    }

    /**
     * Return subject data from Iubenda Consent Solution
     *
     * @param $customerId
     * @return mixed|null
     */
    public function getSubject($customerId) {
        $this->helperData->logDebug("--- START - Client getSubject");

        try {
            //setup
            $this->setCurlOptions();
            $this->addCurlHeaders();

            //params
            $this->helperData->logDebug("------ CustomerId: " . $customerId);
            $url = $this->getHttpApiEndpoint() . $this::SUBJECTS_ENDPOINT . "/" . $customerId;
            $this->curl->get($url);

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ getSubject method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ customerId: " . $customerId);
        }
        $this->helperData->logDebug("--- END - Client getSubject");
        return $response;
    }

    /**
     * Update subject using only $data["subject"] data
     *
     * @param Magento\Customer\Model\Customer $customer
     * @return mixed
     */
    public function updateSubjectFromCustomer(\Magento\Customer\Model\Customer $customer) {
        $this->helperData->logDebug("--- START - Client updateSubjectFromCustomer");

        try {
            //setup
            $this->setCurlOptions();
            $this->curl->setOption(CURLOPT_CUSTOMREQUEST, 'PATCH');
            $this->addCurlHeaders();

            //params
            $data = $this->prepareCustomerData($customer);
            $url = $this->getHttpApiEndpoint() . $this::SUBJECTS_ENDPOINT . "/" . $customer->getId();
            $this->curl->post($url, json_encode($data["subject"]));

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ updateSubjectFromCustomer method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ data: " . print_r($data, true));
        }
        $this->helperData->logDebug("--- END - Client updateSubjectFromCustomer");
        return $response;
    }

    /**
     * Update subject using only $data["subject"] data
     *
     * @param Magento\Newsletter\Model\Subscriber $subscriber
     * @param $subjectId
     * @return mixed
     */
    public function updateSubjectFromSubscriber(\Magento\Newsletter\Model\Subscriber $subscriber, $subjectId) {
        $this->helperData->logDebug("--- START - Client updateSubjectFromSubscriber");

        try {
            //setup
            $this->curl->setOption(CURLOPT_CUSTOMREQUEST, 'PATCH');
            $this->setCurlOptions();
            $this->addCurlHeaders();

            //params
            $data = [
                "subject" => array(
                    "id" => $subscriber->getId(),
                    "email" => $subscriber->getEmail()
                )
            ];
            $url = $this->getHttpApiEndpoint() . $this::SUBJECTS_ENDPOINT . "/" . $subjectId;
            $this->curl->post($url, json_encode($data["subject"]));

            //call
            $response = json_decode($this->curl->getBody(), true);
            $this->helperData->logDebug("------ Response: " . print_r($response, true));
        } catch (Exception $ex) {
            $response = null;
            $this->helperData->log("------ updateSubjectFromSubscriber method");
            $this->helperData->log("------ cUrl Response error: " . $ex->getMessage());
            $this->helperData->log("------ subjectId: " . $subjectId);
            $this->helperData->log("------ data: " . print_r($data, true));
        }
        $this->helperData->logDebug("--- END - Client updateSubjectFromSubscriber");
        return $response;
    }
}
