<?php
/**
 * Downloadable
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Products\Updates;

use Magento\Framework\App\ObjectManagerFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Framework\App\Area;

class Images extends \Drop\Import\Console\Command\Import\Products
{

    /**
     * @var \Drop\Import\Helper\System\Product
     */
    private $helper;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface\Proxy
     */
    private $productRepository;
    /**
     * @var \Magento\Catalog\Model\Product\Gallery\Processor
     */
    protected $imageProcessor;
    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    private $directoryList;
    /**
     * @var \Magento\Theme\Model\Config\Customization
     */
    private $themeCustomizationConfig;
    /**
     * @var \Magento\Eav\Model\Config
     */
    private $eavConfig;
    /**
     * @var \Magento\Eav\Model\Entity\Attribute
     */
    private $attribute;
    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    protected $objectManager;
    /**
     * @var \Magento\Framework\View\ConfigInterface,
     */
    private $viewConfig;
    /**
     * @var \Magento\Theme\Model\ResourceModel\Theme\Collection
     */
    private $themeCollection;
    /**
     * @var \Magento\Catalog\Model\Product\ImageFactory
     */
    private $productImageFactory;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Product\Gallery
     */
    private $productGallery;

    /**
     * Variables
     */
    const IMAGES_COUNTER_SEPARATOR = '_';
    const IMAGES_FORMAT = '.jpg';
    private $productsImages = [];

    /**
     * Images constructor.
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Magento\Framework\App\State $state
     * @param \Drop\Import\Helper\System\Product $helper
     * @param \Magento\Catalog\Api\ProductRepositoryInterface\Proxy $productRepository
     * @param \Magento\Catalog\Model\Product\Gallery\Processor $imageProcessor
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     * @param \Magento\Framework\View\ConfigInterface $viewConfig
     * @param \Magento\Theme\Model\ResourceModel\Theme\Collection $themeCollection
     * @param \Magento\Catalog\Model\Product\ImageFactory $productImageFactory
     * @param \Magento\Theme\Model\Config\Customization $themeCustomizationConfig
     * @param \Magento\Eav\Model\Config $eavConfig
     * @param \Magento\Eav\Model\Entity\Attribute $attribute
     * @param \Magento\Framework\ObjectManagerInterface $objectManager
     * @param \Magento\Catalog\Model\ResourceModel\Product\Gallery $productGallery
     */
    public function __construct(
        ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\Import\Logger\Logger $logger,
        \Magento\Framework\App\State $state,
        \Drop\Import\Helper\System\Product $helper,
        \Magento\Catalog\Api\ProductRepositoryInterface\Proxy $productRepository,
        \Magento\Catalog\Model\Product\Gallery\Processor $imageProcessor,
        \Magento\Framework\App\Filesystem\DirectoryList $directoryList,
        \Magento\Framework\View\ConfigInterface $viewConfig,
        \Magento\Theme\Model\ResourceModel\Theme\Collection $themeCollection,
        \Magento\Catalog\Model\Product\ImageFactory $productImageFactory,
        \Magento\Theme\Model\Config\Customization $themeCustomizationConfig,
        \Magento\Eav\Model\Config $eavConfig,
        \Magento\Eav\Model\Entity\Attribute $attribute,
        \Magento\Framework\ObjectManagerInterface $objectManager,
        \Magento\Catalog\Model\ResourceModel\Product\Gallery $productGallery
    )
    {
        $this->helper = $helper;
        $this->productRepository = $productRepository;
        $this->imageProcessor = $imageProcessor;
        $this->directoryList = $directoryList;
        $this->viewConfig = $viewConfig;
        $this->themeCollection = $themeCollection;
        $this->productImageFactory = $productImageFactory;
        $this->themeCustomizationConfig = $themeCustomizationConfig;
        $this->eavConfig = $eavConfig;
        $this->attribute = $attribute;
        $this->objectManager = $objectManager;
        $this->productGallery = $productGallery;

        parent::__construct($objectManagerFactory, $csvProcessor, $logger, $state);

    }

    protected function configure()
    {
        $this->setName('drop:import:images')
            ->setDescription('Import Images Products');

        parent::configure();
    }

    public function getEntities()
    {
        $this->setEntityCode($this->helper->getEntityCode());
        $this->setBehavior($this->helper->getBehavior());
        $this->setValidationStrategy($this->helper->getValidationStrategy());
        $this->setAllowedErrorCount($this->helper->getAllowedErrorCount());
        $this->setIgnoreDuplicates($this->helper->getIgnoreDuplicates());

        $importMethod = $this->helper->getImportMethod();
        if ($importMethod != \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_LOCAL_FILE_PATH) {
            //Download file
            die('TODO');
        }

        $fileName = $this->helper->getImportPathUrl();
        $csvIterationObject = $this->readCSV(BP . '/' . $fileName);

        $i = -1;
        foreach ($csvIterationObject as $row) {
            if(empty($row['sku'])) {
                echo PHP_EOL . "SKU was not found. Skipping.";
                continue;
            }

            $row['sku'] = strtolower($row['sku']);

            try {
                $product = $this->productRepository->get($row['sku']);
            } catch (NoSuchEntityException $e) {
                echo PHP_EOL . strtoupper($row['sku']) . ' NoSuchEntityException on Other Attributes: ' . $e->getMessage() . '. Skipping.';
                continue;
            } catch (\Exception $e) {
                echo PHP_EOL . strtoupper($row['sku']) . ' Exception on Other Attributes: ' . $e->getMessage() . '. Skipping.';
                continue;
            }

            // Images
            $imagesAbsolutePath = $this->directoryList->getPath('media') . '/import/images/';
            $imagesRelativePath = 'images/';
            $configurableImages = [];
            for($imagesCounter=1; $imagesCounter<10; $imagesCounter++) {
                $imageName = strtoupper($row['sku']) . self::IMAGES_COUNTER_SEPARATOR . '0' . $imagesCounter . self::IMAGES_FORMAT;
                if(file_exists($imagesAbsolutePath . $imageName)) {
                    if(!isset($configurableImages['additional_images'])) {
                        $configurableImages['additional_images'] = $imagesRelativePath . $imageName;
                    } else {
                        $configurableImages['additional_images'] .= ',' . $imagesRelativePath . $imageName;
                    }
                    foreach($this->getImageRoleKey($imagesCounter) as $imageRole) {
                        $configurableImages[$imageRole] = $imagesRelativePath . $imageName;
                    }
                }
            }

            if(!empty($configurableImages)) {
                $i++;
                $this->productsImages[$i] = array_merge([
                    'sku' => $row['sku']
                ], $configurableImages);
            }


            if ($product->getTypeId() == "configurable"):
                $productId = $product->getId();
                $existingMediaGalleryEntries = $product->getMediaGalleryEntries();
                if (is_array($existingMediaGalleryEntries)) {
                    foreach ($existingMediaGalleryEntries as $key => $entry) {
                        unset($existingMediaGalleryEntries[$key]);
                        $image = $entry->getFile();
                        $this->imageProcessor->removeImage($product, $image);
                        $image = $this->directoryList->getPath('media').'/catalog/product' . $image;
                        if (file_exists($image)) {
                            unlink($image);
                        }
                        $gallery = $product->getMediaGalleryImages();
                        if (count($gallery) > 0) {
                            foreach($gallery as $image){
                                $this->productGallery->deleteGallery($image->getValueId());
                            }
                        }
                    }
                }

                $product->setMediaGalleryEntries($existingMediaGalleryEntries);
                try {
                    $product = $this->productRepository->save($product);
                } catch (\Exception $e) {
                    $this->logger->info(': ' . $e->getMessage());
                }

                $simpleProducts = $this->getSimpleAssociated($product);

                foreach ($simpleProducts as $simpleProduct) {

                    $simpleSku = $simpleProduct->getSku();
                    if (strpos($simpleSku, $product->getSku()) !== false) {
                        $existingMediaGalleryEntries = $simpleProduct->getMediaGalleryEntries();
                        if (is_array($existingMediaGalleryEntries)) {
                            foreach ($existingMediaGalleryEntries as $key => $entry) {
                                unset($existingMediaGalleryEntries[$key]);
                                $image = $entry->getFile();
                                $this->imageProcessor->removeImage($simpleProduct, $image);
                                $image = $this->directoryList->getPath('media').'/catalog/product' . $image;
                                if (file_exists($image)) {
                                    unlink($image);
                                }
                                $gallery = $simpleProduct->getMediaGalleryImages();
                                if (count($gallery) > 0) {
                                    foreach($gallery as $image){
                                        $this->productGallery->deleteGallery($image->getValueId());
                                    }
                                }
                            }
                        }

                        $simpleProduct->setMediaGalleryEntries($existingMediaGalleryEntries);
                        try {
                            $simpleProduct = $this->productRepository->save($simpleProduct);
                        } catch (\Exception $e) {
                            $this->logger->info(': ' . $e->getMessage());
                        }
                    }
                    //Swatches Images
                    $simpleImages = [];
                    for($imagesCounter=1; $imagesCounter<10; $imagesCounter++) {
                        //Image name is sku without size part (-42)
                        $imageName = strtoupper(substr($simpleSku, 0, strrpos($simpleSku, '-'))) . self::IMAGES_COUNTER_SEPARATOR . '0' . $imagesCounter . self::IMAGES_FORMAT;
                        if(file_exists($imagesAbsolutePath . $imageName)) {
                            if(!isset($simpleImages['additional_images'])) {
                                $simpleImages['additional_images'] = $imagesRelativePath . $imageName;
                            } else {
                                $simpleImages['additional_images'] .= ',' . $imagesRelativePath . $imageName;
                            }
                            foreach($this->getImageRoleKey($imagesCounter) as $imageRole) {
    //                            echo PHP_EOL . 'SIMPLE' . $simpleSku . ' - ' . $imageRole . ': ' . $imagesRelativePath . $imageName;
                                $simpleImages[$imageRole] = $imagesRelativePath . $imageName;
                            }
                        }
                    }


                    if(!empty($simpleImages) && !empty($simpleSku)) {
                        $i++;
                        $this->productsImages[$i] = array_merge([
                            'sku' => $simpleSku
                        ], $simpleImages);
                    }

                }
            endif;
        }

//        echo "<br/><pre>";
//        print_r($this->productsImages);
//        echo "</pre><br/>";
//        die('<br/>' . __METHOD__ . '1');

        return $this->productsImages;

    }

    /**
     * @param $i
     * @return string
     */
    public function getImageRoleKey($i) {
        switch ($i) {
            case 1:
                return ['base_image','small_image','thumbnail_image'];
            case 2:
                return ['hover_image'];
        }
        return [];
    }

    public function afterFinishImport(){
        $themes = $this->getThemesInUse();
        $viewImages = $this->getViewImages($themes->getItems());

        echo 'Regenerating catalog image caches for ' .  count($this->productsImages) . ' products and ' . count($viewImages) . ' views...';

        foreach($this->productsImages as $productImages) {
            $sku = $productImages['sku'];
            $product = $this->productRepository->get($productImages['sku']);
            $productMediaGallery = $product->getMediaGallery();
            if(empty($productMediaGallery['images'])) {
                continue;
            }

            foreach($productMediaGallery['images'] as $image) {
                if(empty($image['file'])) {
                    echo PHP_EOL . 'Image file does not exist for product ' . $sku;
                }

                $originalImageName = $image['file'];
                foreach ($viewImages as $viewImage) {
                    try {
                        $image = $this->makeImage($originalImageName, $viewImage);
                        $image->resize();
                        $image->saveFile();
                    } catch (\Exception $e) {
                        echo PHP_EOL . 'EXCEPTION making image: ' . $e->getMessage();
                    }
                }
            }
        }

        echo PHP_EOL . PHP_EOL . 'Catalog product image cache regenerated correctly' . PHP_EOL. PHP_EOL;
    }

    protected function getThemesInUse()
    {
        $themesIdsInUse = [];
        $registeredThemes = $this->loadRegisteredThemes();
        $storesByThemes   = $this->themeCustomizationConfig->getStoresByThemes();
        foreach ($registeredThemes as $registeredTheme) {
            if (array_key_exists($registeredTheme->getThemeId(), $storesByThemes) || $registeredTheme->getCode() == 'Magento/backend') {
                $themesIdsInUse[] = $registeredTheme->getThemeId();
            }
        }

        $resource = $this->objectManager->get(\Magento\Framework\App\ResourceConnection::class);

        $catalogProductEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Product::ENTITY)->getId();
        $productCustomDesignAttributeId = $this->attribute->loadByCode($catalogProductEntityId, 'custom_design')->getId();

        $productSql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_product_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $productCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $productThemeIds = $resource->getConnection()->fetchCol($productSql);
        foreach ($productThemeIds as $productThemeId) {
            if (array_key_exists($productThemeId, $storesByThemes) && !in_array($productThemeId, $themesIdsInUse)) {
                $themesIdsInUse[] = $productThemeId;
            }
        }

        $catalogCategoryEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Category::ENTITY)->getId();
        $categoryCustomDesignAttributeId = $this->attribute->loadByCode($catalogCategoryEntityId, 'custom_design')->getId();

        $categorySql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_category_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $categoryCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $categoryThemeIds = $resource->getConnection()->fetchCol($categorySql);
        foreach ($categoryThemeIds as $categoryThemeId) {
            if (array_key_exists($categoryThemeId, $storesByThemes)
                && !in_array($categoryThemeId, $themesIdsInUse) ) {
                $themesIdsInUse[] = $categoryThemeId;
            }
        }

        $pageSql = $resource->getConnection()
            ->select()
            ->from(
                ['page' => $resource->getTableName('cms_page')],
                ['custom_theme']
            )
            ->where('custom_theme > 0')
            ->group('custom_theme');

        $pageThemeIds = $resource->getConnection()->fetchCol($pageSql);
        foreach ($pageThemeIds as $pageThemeId) {
            if (array_key_exists($pageThemeId, $storesByThemes)
                && !in_array($pageThemeId, $themesIdsInUse) ) {
                $themesIdsInUse[] = $pageThemeId;
            }
        }
        foreach ($registeredThemes as $registeredTheme) {
            if (!in_array($registeredTheme->getThemeId(), $themesIdsInUse)) {
                $registeredThemes->removeItemByKey($registeredTheme->getThemeId());
            }
        }

        return $registeredThemes;
    }

    public function loadRegisteredThemes()
    {
        $this->themeCollection->clear();
        return $this->themeCollection->setOrder('theme_title', \Magento\Framework\Data\Collection::SORT_ORDER_ASC)
            ->filterVisibleThemes();
    }

    /**
     * Get view images data from themes
     * @param array $themes
     * @return array
     */
    private function getViewImages(array $themes): array
    {
        $viewImages = [];
        foreach ($themes as $theme) {
            $config = $this->viewConfig->getViewConfig([
                'area' => Area::AREA_FRONTEND,
                'themeModel' => $theme,
            ]);
            $images = $config->getMediaEntities('Magento_Catalog', ImageHelper::MEDIA_TYPE_CONFIG_NODE);
            foreach ($images as $imageId => $imageData) {
                $uniqIndex = $this->getUniqueImageIndex($imageData);
                $imageData['id'] = $imageId;
                $viewImages[$uniqIndex] = $imageData;
            }
        }
        return $viewImages;
    }

    /**
     * Make image
     * @param string $originalImagePath
     * @param array $imageParams
     * @return \Drop\Bettercatalogimageresize\Plugin\Product\ImagePlugin
     * @throws \Exception
     */
    private function makeImage(string $originalImagePath, array $imageParams)
    {
        /** @var \Drop\Bettercatalogimageresize\Plugin\Product\ImagePlugin $image */
        $image = $this->productImageFactory->create();

        if (isset($imageParams['height'])) {
            $image->setHeight($imageParams['height']);
        }
        if (isset($imageParams['width'])) {
            $image->setWidth($imageParams['width']);
        }
        if (isset($imageParams['aspect_ratio'])) {
            $image->setKeepAspectRatio($imageParams['aspect_ratio']);
        }
        if (isset($imageParams['frame'])) {
            $image->setKeepFrame($imageParams['frame']);
        }
        if (isset($imageParams['transparency'])) {
            $image->setKeepTransparency($imageParams['transparency']);
        }
        if (isset($imageParams['constrain'])) {
            $image->setConstrainOnly($imageParams['constrain']);
        }
        if (isset($imageParams['background'])) {
            $image->setBackgroundColor($imageParams['background']);
        }

        $image->setDestinationSubdir($imageParams['type']);
        $image->setBaseFile($originalImagePath);
        return $image;
    }

    /**
     * Get unique image index
     * @param array $imageData
     * @return string
     */
    private function getUniqueImageIndex(array $imageData): string
    {
        ksort($imageData);
        unset($imageData['type']);
        return md5(json_encode($imageData));
    }

}
