<?php
/**
 * Configurable
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Products;

use Magento\Framework\App\ObjectManagerFactory;

class Configurable extends \Drop\Import\Console\Command\Import\Products
{

    const IMAGES_COUNTER_SEPARATOR = '_';
    //TODO: Dynamic
    const IMAGES_FORMAT = '.jpg';
    //TODO: Dynamic
    const DEFAULT_QTY = '0';

    /**
     * Product Helper: \Drop\Import\Helper\System\Product
     * @var type
     */
    private $helper;
    /**
     * AttributeFactory
     * @var type
     */
    private $attributeFactory;
    /**
     * @var \Magento\Eav\Api\AttributeSetRepositoryInterface
     */
    private $attributeSet;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;
    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    private $directoryList;
    /**
     * @var \Magento\Catalog\Model\Product\UrlFactory
     */
    private $urlFactory;

    /**
     * Attribute setted on this variable will be excluded from being setted to the configurable. Will be set to the simple associated products instead.
     * Override this variable on custom script to choose these attributes.
     * @var
     */
    public $setAttributeOnSimpleExcludingFromConfigurable = [];

    /**
     * Products constructor.
     * @param ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\System\Product $helper
     * @param AttributeFactory $attributeFactory
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\Catalog\Model\Product\UrlFactory $urlFactory
     * @param \Magento\Eav\Api\AttributeSetRepositoryInterface $attributeSet
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     */
    public function __construct(
        ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\Import\Logger\Logger $logger,
        \Magento\Framework\App\State $state,
        \Drop\Import\Helper\System\Product $helper,
        \Drop\Import\Model\AttributeFactory $attributeFactory,
        \Magento\Catalog\Api\ProductRepositoryInterface\Proxy $productRepository,
        \Magento\Catalog\Model\Product\UrlFactory $urlFactory,
        \Magento\Eav\Api\AttributeSetRepositoryInterface\Proxy $attributeSet,
        \Magento\Framework\App\Filesystem\DirectoryList $directoryList
    )
    {
        $this->helper = $helper;
        $this->attributeFactory = $attributeFactory;
        $this->productRepository = $productRepository;
        $this->attributeSet = $attributeSet;
        $this->directoryList = $directoryList;
        $this->urlFactory = $urlFactory;
        parent::__construct($objectManagerFactory, $csvProcessor, $logger, $state);
    }

    protected function configure()
    {
        $this->setName('drop:import:configurable')
            ->setDescription('Import Configurable Products');
        $this->setEntityCode($this->helper->getEntityCode());
        $this->setBehavior($this->helper->getBehavior());
        $this->setValidationStrategy($this->helper->getValidationStrategy());
        $this->setAllowedErrorCount($this->helper->getAllowedErrorCount());
        $this->setIgnoreDuplicates($this->helper->getIgnoreDuplicates());
        $this->setImportImagesFileDir($this->helper->getImportImagesFileDir());
        parent::configure();
    }

    /**
     * SAMPLES:
     * $products [] = array(
     * 'sku' => "SIMPLE-RED-MIDDLE",
     * 'attribute_set_code' => 'Default',
     * 'product_type' => 'simple',
     * 'product_websites' => 'base',
     * 'name' => 'FireGento Simple Product Red,Size Middle',
     * 'price' => '1.0000',
     * 'color' => 'red',
     * 'size' => 'M',
     * 'multiselect' => 'Test|Test3'
     * //            'visibility' => 'Catalog, Search',
     * //            'tax_class_name' => 'Taxable Goods',
     * //            'product_online' => '1',
     * //            'weight' => '1.0000',
     * //            'short_description' => NULL,
     * //            'description' => '',
     * );
     * $products [] = array(
     * 'sku' => 'CONFIG-Product',
     * 'attribute_set_code' => 'Default',
     * 'product_type' => 'configurable',
     * 'product_websites' => 'base',
     * 'name' => 'FireGento Test Product Configurable',
     * 'price' => '10.000',
     * 'configurable_variation_labels' => 'Color',
     * 'configurable_variations' => array(
     * array(
     * 'sku' => 'SIMPLE-RED-MIDDLE',
     * 'color' => 'red',
     * 'size' => 'M'),
     * )
     * );
     * @return array
     *
     * il prodotto viene creato con stato DISABLED e visibility SEARCH
     * al momento del caricamento prezzi viene impostata la visibilità CATALOG/SEARCH
     * al momento dell'aggiornamento dello stock (eventualmente) stato ENABLED
     */
    protected function getEntities()
    {

        $importMethod = $this->helper->getImportMethod();
        if ($importMethod != \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_LOCAL_FILE_PATH) {
            //Download file
            die('TODO');
        }

        $fileName = $this->helper->getImportPathUrl();
        $csvIterationObject = $this->readCSV(BP . '/' . $fileName);
        $languages = $this->getLanguages();

        $products = [];
        $i = 0;
        foreach ($csvIterationObject as $row) {

            if (empty($row['sku'])) {
                echo PHP_EOL . "SKU was not found. Skipping.";
                continue;
            }

            $row['sku'] = strtolower($row['sku']);
            $row['attribute_set_code'] = strtolower($row['attribute_set_code']);

            if (empty($row['name_' . self::DEFAULT_STORE])) {
                echo PHP_EOL . "NAME was not found for product: " . strtoupper($row['sku']) . '. Skipping.';
                continue;
            }
            if (empty($row['categories'])) {
                echo PHP_EOL . "CATEGORIES was not found for product: " . strtoupper($row['sku']) . '. Skipping.';
                continue;
            }
            if (empty($row['store'])) {
                echo PHP_EOL . "STORES was not found for product: " . strtoupper($row['sku']) . '. Skipping.';
                continue;
            }

            $websites = $row['store'];
            if (strpos($row['store'], '|') != false) {
                $websites = str_replace('|', ',', $row['store']);
            }

            // Images
            $imagesAbsolutePath = $this->directoryList->getPath('media') . '/import/images/';
            $imagesRelativePath = 'images/';
            $images = [];
            for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                $imageName = strtoupper($row['sku']) . self::IMAGES_COUNTER_SEPARATOR . '0' . $imagesCounter . self::IMAGES_FORMAT;
                if (file_exists($imagesAbsolutePath . $imageName)) {
                    if (!isset($images['additional_images'])) {
                        $images['additional_images'] = $imagesRelativePath . $imageName;
                    } else {
                        $images['additional_images'] .= ',' . $imagesRelativePath . $imageName;
                    }
                    foreach ($this->getImageRoleKey($imagesCounter) as $imageRole) {
//                        echo PHP_EOL . 'CONFIGURABLE: ' . strtoupper($row['sku']) . ' - ' . $imageRole . ': ' . $imagesRelativePath . $imageName;
                        $images[$imageRole] = $imagesRelativePath . $imageName;
                    }
                }
            }

            //Images doesn't exist. Skipping product import
            if (empty($images)) {
                echo PHP_EOL . "IMAGE was not found for product: " . strtoupper($row['sku']) . '. Skipping.';
                continue;
            }

            $configurableVariations = [];
            $superAttributeOptions = $this->attributeFactory->create()->getSuperAttributeOptions($row['attribute_set_code']);
            foreach ($superAttributeOptions as $superAttributeCode => $simpleOptions) {
                //Simple associated
                foreach ($simpleOptions as $simpleOption) {

                    if (empty($simpleOption['value']) || empty($simpleOption['label'])) {
                        continue;
                    }

                    $products[$i] = [
                        'sku'                => $row['sku'] . '-' . $simpleOption['label'],
                        'store_view_code'    => '',
                        'attribute_set_code' => $row['attribute_set_code'],
                        'product_type'       => 'simple',
                        'product_websites'   => $websites,
                        'name'               => $row['name_' . self::DEFAULT_STORE] . ' - ' . $row['attribute_set_code'] . ': ' . $simpleOption['label'],
                        $superAttributeCode  => $simpleOption['label'],
                        'categories'         => 'Default Category/' . $row['categories'],
                        'price'              => '0',
                        'weight'             => '1.0000',
                        'visibility'         => 'Not Visible Individually',
                        'status'             => 'Disabled'
                    ];
                    $configurableVariations[$i] = [
                        'sku'               => $products[$i]['sku'],
                        $superAttributeCode => $products[$i][$superAttributeCode]
                    ];

                    $i++;

                }
            }

            $products[$i] = [
                'sku'                => $row['sku'],
                'store_view_code'    => '',
                'attribute_set_code' => $row['attribute_set_code'], //TODO: Check if exist
                'product_type'       => 'configurable',
                'product_websites'   => $websites,
                'categories'         => 'Default Category/' . $row['categories'],
                'name'               => $row['name_' . self::DEFAULT_STORE],
                'description'        => $this->convertStringToUtf8($row['description_' . self::DEFAULT_STORE]),
                'price'              => '0',
                'url_key'            => $this->urlFactory->create()->formatUrlKey($this->formatProductUrl($row['name_' . self::DEFAULT_STORE] . '-' . $row['sku'])),
                'meta_title'         => $row['name_' . self::DEFAULT_STORE] . ' - ' . $row['sku'],
                'meta_description'   => $this->convertStringToUtf8(substr($row['name_' . self::DEFAULT_STORE] . ' - ' . $row['sku'] . ' - ' . $row['description_' . self::DEFAULT_STORE], 0, 84)),
                'visibility'         => 'Search',
                'status'             => 'Disabled',
                'manage_stock'       => 0
            ];

            if (!empty($images)) {
                $products[$i] = array_merge($products[$i], $images);
            }

            //Other attributes
            $products[$i] = array_merge($products[$i], $this->getConfigurableAttributes($row, self::DEFAULT_STORE));

            //Simple products associated
            foreach ($configurableVariations as $configurableVariation) {
                $products[$i]['configurable_variations'][] = $configurableVariation;
            }

            //Translation rows
            if ($translations = $this->getLanguagesVariables($languages, $row, $i)) {
                foreach ($translations as $translation) {
                    array_push($products, $translation);
                    $i++;
                }
            }

            $i++;
        }

        echo "\n<pre>";
        print_r($products);
        echo "<pre>\n";
        die(__METHOD__ . '1');

        return $products;
    }

    /**
     * @param $i
     * @return string
     */
    public function getImageRoleKey($i)
    {
        switch ($i) {
            case 1:
                return ['base_image', 'small_image', 'thumbnail_image'];
            case 2:
                return ['hover_image'];
        }
        return [];
    }

    /**
     * @param $languages
     * @param $row
     * @param $i
     * @return array
     */
    public function getLanguagesVariables($languages, $row, $i)
    {
        $products = [];
        $langCounter = 0;
        foreach ($languages as $lang) {
            if ($lang == self::DEFAULT_STORE) {
                continue;
            }
            $i++;
            $products[$i] = [
                'sku'                => $row['sku'],
                'store_view_code'    => $lang,
                'attribute_set_code' => $row['attribute_set_code'], //TODO: Check if exist
                'name'               => $row['name_' . $lang],
                'description'        => $this->convertStringToUtf8($row['description_' . $lang]),
                'url_key'            => $this->urlFactory->create()->formatUrlKey($row['name_' . $lang] . '-' . $row['sku']),
                'meta_title'         => $row['name_' . $lang] . ' - ' . $row['sku'],
                'meta_description'   => $this->convertStringToUtf8(substr($row['name_' . $lang] . ' - ' . $row['sku'] . ' - ' . $row['description_' . $lang], 0, 84))
            ];
            //Other attributes
            $products[$i] = array_merge($products[$i], $this->getConfigurableAttributes($row, $lang));
            $langCounter++;
        }

        return $products;
    }

    /**
     * @param $row
     * @param $storeView
     * @return array
     */
    public function getConfigurableAttributes($row, $storeView)
    {
        $header = $this->getCsvHeader();
        $attributes = [];
        $configurableAttributes = $this->attributeFactory->create()->getUserDefinedAttributeByAttributeSet($row['attribute_set_code']);
        foreach ($header as $field) {
            if (in_array($field, $configurableAttributes) && !empty($row[$field]) && !$this->setAttributeOnSimpleExcludingFromConfigurable($field)) {
                $attributes[$field] = $row[$field];
            }
            if (
                (strpos($field, '_' . $storeView) !== false) &&
                in_array(str_replace('_' . $storeView, '', $field), $configurableAttributes) && !empty($row[$field]) &&
                !$this->setAttributeOnSimpleExcludingFromConfigurable($field)
            ) {
                $attributes[str_replace('_' . $storeView, '', $field)] = $row[$field];
            }
        }

        return $attributes;
    }

}
