<?php

namespace Drop\Import\Console\Command\Import;

use Magento\Framework\App\ObjectManagerFactory;
use Drop\Import\Console\Command\AbstractImportCommand;

class Customer extends AbstractImportCommand
{
    /**
     * Customer Helper: \Drop\Import\Helper\System\Customer
     * @var type 
     */
    private $helper;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;

    private $baseWebsiteCode;
    private $defaultStoreCode;
    private $defaultStoreLabel;
    private $generalGroupId;
//    private $genderConversion;
    public $genderConversion = [
        'Uomo' => 'Male',
        'Donna' => 'Female',
    ];
//    public $useGeocode = 0;
    public $useGeocode = 1;
    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    private $dataObjectFactory;
    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    private $customerFactory;
    /**
     * @var \Magento\Directory\Model\RegionFactory
     */
    private $regionFactory;


    /**
     * Customer constructor.
     * @param ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Magento\Framework\App\State $state
     * @param \Magento\Framework\ObjectManager\ConfigLoaderInterface $configLoader
     * @param \Drop\Import\Helper\System\Customer $helper
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Framework\DataObjectFactory $dataObjectFactory
     * @param \Magento\Customer\Model\CustomerFactory $customerFactory
     * @param \Magento\Directory\Model\RegionFactoryFactory $regionFactory
     */
    public function __construct(
        ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,  
        \Drop\Import\Logger\Logger $logger,
        \Magento\Framework\App\State $state,
        \Drop\Import\Helper\System\Customer $helper,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\DataObjectFactory $dataObjectFactory,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Magento\Directory\Model\RegionFactory $regionFactory
    )
    {
        $this->helper = $helper;
        $this->date = $date;
        $this->dataObjectFactory = $dataObjectFactory;
        $this->customerFactory = $customerFactory;
        $this->regionFactory = $regionFactory;
        parent::__construct($objectManagerFactory, $csvProcessor, $logger, $state);
    }

    protected function getEntities() {}

    /**
     * Reindex after
     */
    public function afterFinishImport(){
        $indexerFactory = $this->objectManager->create('Magento\Indexer\Model\IndexerFactory');
        $indexer = $indexerFactory->create();
        $indexer->load("customer_grid");
        $indexer->reindexAll();
    }

    /**
     * Website code calculation or fallback to base website. Extend this to introduce custom calculation
     * @param null $websiteCode
     * @return mixed|null
     */
    public function getWebsiteCode($websiteCode = null) {
        if(empty($websiteCode)) {
            if(!isset($this->baseWebsiteCode)) {
                $this->baseWebsiteCode = $this->helper->getBaseWebsiteCode();
            }
            return $this->baseWebsiteCode;
        }
        return $websiteCode;
    }

    /**
     * Website id calculation or fallback to base website id. Extend this to introduce custom calculation
     * @param null $websiteId
     * @return mixed|null
     */
    public function getWebsiteId($websiteId = null) {
        if(empty($websiteId)) {
            if(!isset($this->baseWebsiteId)) {
                $this->baseWebsiteId = $this->helper->getBaseWebsiteId();
            }
            return $this->baseWebsiteId;
        }
        return $websiteId;
    }

    /**
     * Store id calculation or fallback to default store id. Extend this to introduce custom calculation
     * @param null $storeId
     * @return null
     */
    public function getStoreId($storeId = null) {
//        $country = $storeId;
//        $store = $this->helper->getStoreByCountry($country);
//        if($store && !empty($store->getStoreId())) {
//            return $store->getStoreId();
//        }

        if(empty($storeId)) {
            if(!isset($this->defaultStoreId)) {
                $this->defaultStoreId = $this->helper->getDefaultStoreId();
            }
            return $this->defaultStoreId;
        }

        $country = $storeId;
        if($country == 'IT') {
            return 1;
        }
        return 2;

        return $storeId;
    }

    /**
     * Store code calculation or fallback to default store code. Extend this to introduce custom calculation
     * @param null $storeCode
     * @return mixed|null
     */
    public function getStoreCode($storeCode = null) {
//        $country = $storeCode;
//        $store = $this->helper->getStoreByCountry($country);
//        if($store && !empty($store->getCode())) {
//            return $store->getCode();
//        }

        if(empty($storeCode)) {
            if(!isset($this->defaultStoreCode)) {
                $this->defaultStoreCode = $this->helper->getDefaultStoreCode();
            }
            return $this->defaultStoreCode;
        }

        $country = $storeCode;
        if($country == 'IT') {
            return 'it';
        }
        return 'en';

        return $storeCode;
    }

    /**
     * Store label calculation or fallback to default store label. Extend this to introduce custom calculation
     * @param null $storeLabel
     * @return mixed|null
     */
    public  function getStoreLabel($storeLabel = null) {
        if(empty($storeLabel)) {
            if(!isset($this->defaultStoreLabel)) {
                $this->defaultStoreLabel = $this->helper->getDefaultStoreLabel();
            }
            return $this->defaultStoreLabel;
        }

        $country = $storeLabel;
        if($country == 'IT') {
            return 'Italiano';
        }
        return 'English';

        return $storeLabel;
    }

    /**
     * Group id calculation or fallback to General group id. Extend this to introduce custom calculation
     * @param null $groupId
     * @return mixed|null
     */
    public function getGroupId($groupId = null) {
        if(empty($groupId)) {
            if(!isset($this->generalGroupId)) {
                $this->generalGroupId = $this->helper->getGeneralGroupId();
            }
            return $this->generalGroupId;
        }
        return $groupId;
    }

    public function genderConversion($csvGender) {
        if(empty($csvGender)) {
            return null;
        }
        if(isset($this->genderConversion[$csvGender])) {
            return $this->genderConversion[$csvGender];
        }
        echo PHP_EOL . "Gender conversion not found for value: " . $csvGender;
        return null;
    }

    /**
     * Date conversion. Extend this to introduce custom conversion
     * @param $csvDate
     * @return mixed
     */
    public function dateConversion($csvDate) {
        if(empty($csvDate)) {
            return;
        }
        if(strpos($csvDate, '-') !== false) {
            $delimiter = '-';
        } elseif(strpos($csvDate, '/') !== false) {
            $delimiter = '/';
        } elseif(strpos($csvDate, '.') !== false) {
            $delimiter = '.';
//        } elseif(strpos($csvDate, ' ') !== false) {
            //How do detect 1 agosto 1963?
//            $delimiter = ' ';
        } else {
            return;
        }

        $fields = explode($delimiter, $csvDate);
        if(empty($fields[0]) || empty($fields[1]) || empty($fields[2])) {
            return false;
        }

        if(strlen((string)$fields[2]) == 2) {
            $dt = \DateTime::createFromFormat('y', $fields[2]);
            $fields[2] = $dt->format('Y');
            $csvDate = implode($delimiter, $fields);
        }

        if(!$this->checkDateIntegrity($fields)) {
            return;
        }

        //Solo per date in formato italiano. How to detect date format before?
        return str_replace($delimiter, '-', $csvDate);

    }

    public  function checkDateIntegrity(array $dateArr) {
        $d = $dateArr[0];
        $m = $dateArr[1];
        $y = $dateArr[2];

        if(($y < 1901) || ($y > date("Y"))) {
            return false;
        }

        //Days and month //TODO: Check if date is American or Italian
        if(($d > 31) || ($m > 12)) {
            return false;
        }

        return true;
    }

    public function dobConversion($csvDate) {
        return $this->dateConversion($csvDate);
    }

    public function geocodeModuleExist() {
        if (class_exists(\Limesharp\Stockists\Controller\Adminhtml\Stores\Geocode::class) &&
            class_exists(\Limesharp\Stockists\Model\Geocode::class)) {
            return true;
        }
        return false;
    }

    public function formatAddress($data) {
        $addressObj = $this->dataObjectFactory->create();
        $addressObj->setAddress($data['indirizzo'])
            ->setCity($data['localita'])
            ->setRegion($data['provincia'])
            ->setPostcode($data['cap']);
//            ->getCountry($data['indirizzo']);
        $geocodeController = $this->objectManager->get('Limesharp\Stockists\Controller\Adminhtml\Stores\Geocode');
        return $geocodeController->getStoreFormattedAddress($addressObj);
    }

    public function geocodeAddress($address) {
        if(empty($address)) {
            return false;
        }

        $geocodeModel = $this->objectManager->get('Limesharp\Stockists\Model\Geocode');
        try {
            return $geocodeModel->completeGeocode($address);
        } catch (\Exception $e) {
            echo PHP_EOL . $e->getMessage();
            return false;
        }
    }

    public function getIfCustomerExistByEmail($email) {
        if(empty($email)) {
            return false;
        }

        $customer = $this->customerFactory->create()->loadByEmail($email);
        if ($customer->getId()) {
            return true;
        }
        return false;
    }

    /**
     * Cache problems!! Need to use object manager to istanciate class every time or in case of error it will return the latest data
     * @param $regionCode
     * @param $countryId
     * @return bool
     */
    public function getRegionExist($regionCode, $countryId) {
        $region = $this->regionFactory->create()->loadByCode($regionCode,$countryId);
        if($region && $region->getRegionId()) {
            return true;
        }
        return false;
    }

}



