<?php

namespace Drop\Import\Console\Command;

use Magento\Backend\App\Area\FrontNameResolver;
use Magento\Framework\App\ObjectManager\ConfigLoader;
use Magento\Framework\App\ObjectManagerFactory;
use Magento\Framework\App\State;
use Magento\ImportExport\Model\Import;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManager;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

abstract class AbstractImportCommand extends Command
{

    /**
     * @var string
     */
    protected $behavior;
    /**
     * @var string
     */
    protected $entityCode;
    /**
     * @var string
     */
    protected $ignoreDuplicates;
    /**
     * @var string
     */
    protected $importImagesFileDir;
    /**
     * @var string
     */
    protected $validationStrategy;
    /**
     * @var string
     */
    protected $allowedErrorCount;
    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    protected $objectManager;
    /**
     * Object manager factory
     *
     * @var ObjectManagerFactory
     */
    private $objectManagerFactory;
    /**
     * @var \Magento\Framework\File\Csv
     */
    protected $csvProcessor;
    /**
     * @var \Drop\Import\Logger\Logger
     */
    protected $logger;

    public $csvHeader = [];
    public $optionsArgument = [];

    const CSV_DELIMITER = ',';

    //TODO: RENDERE DINAMICA?!
    const DEFAULT_STORE = 'en';
    /**
 * @var Magento\Framework\App\State
 */
    private $state;

    /**
     * Constructor
     *
     * @param ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\Import\Logger\Logger $logger
     * @param Magento\Framework\App\State $state
     */
    public function __construct(
        ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\Import\Logger\Logger $logger,
        \Magento\Framework\App\State $state
    ) {
        $this->objectManagerFactory = $objectManagerFactory;
        $this->csvProcessor = $csvProcessor;
        $this->logger = $logger;
        $this->state = $state;

        parent::__construct();
    }

    public function arrayToAttributeString($array)
    {
        $attributes_str = NULL;
        foreach ($array as $attribute => $value) {
            $attributes_str .= "$attribute=$value,";
        }
        return $attributes_str;
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Magento\Framework\Exception\LocalizedException
     * @return null|int null or 0 if everything went fine, or an error code
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        echo PHP_EOL; // Fix Area code error
        try {
            $this->state->getAreaCode();
        } catch (\Exception $e) {
            $this->state->setAreaCode(\Magento\Framework\App\Area::AREA_ADMINHTML);
        }

        $output->writeln('Import started');

        $time = microtime(true);

        /** @var \FireGento\FastSimpleImport\Model\Importer $importerModel */
        $omParams = $_SERVER;
        $omParams[StoreManager::PARAM_RUN_CODE] = 'admin';
        $omParams[Store::CUSTOM_ENTRY_POINT_PARAM] = true;
        $this->objectManager = $this->objectManagerFactory->create($omParams);
        $importerModel = $this->objectManager->create('FireGento\FastSimpleImport\Model\Importer');

        $definitionOptions = $this->getDefinition()->getOptions();

        $options = [];
        foreach($definitionOptions as $option) {
            $optionValue = $input->getOption($option->getName());
            if($optionValue) {
                $options[$option->getName()] = $optionValue;
            }
        }

        $this->setOptionsArgument($options);

        $productsArray = $this->getEntities();

        $importerModel->setBehavior($this->getBehavior());
        $importerModel->setEntityCode($this->getEntityCode());
        $importerModel->setValidationStrategy($this->getValidationStrategy());
        $importerModel->setAllowedErrorCount($this->getAllowedErrorCount());
        $importerModel->setIgnoreDuplicates($this->getIgnoreDuplicates());
        $importerModel->setImportImagesFileDir($this->getImportImagesFileDir());
        $adapterFactory = $this->objectManager->create('FireGento\FastSimpleImport\Model\Adapters\NestedArrayAdapterFactory');
        $importerModel->setImportAdapterFactory($adapterFactory);

        try {
            $result = $importerModel->processImport($productsArray);
            echo "\n" . $result . "\n";
        } catch (\Exception $e) {
            $this->logger->info($e->getMessage());
            $output->writeln($e->getMessage());
        }

        $output->write($importerModel->getLogTrace());
        $output->write($importerModel->getErrorMessages());

        $output->writeln('Import finished. Elapsed time: ' . round(microtime(true) - $time, 2) . 's' . "\n");
        $this->afterFinishImport();

    }

    /**
     * @return array
     */
    abstract protected function getEntities();

    /**
     * Read CSV file
     * @param type $fileName
     * @return type
     */
    protected function readCSV($fileName)
    {
        $data = $this->csvProcessor->getData($fileName);
        $this->csvHeader = array_shift($data);

        $i = 0;
        $formattedData = [];
        foreach ($data as $row) {
            foreach ($row as $key => $value) {
                if(!isset($this->csvHeader[$key])){continue;}
                $formattedData[$i][$this->csvHeader[$key]] = str_replace("’", "'", $value);
            }
            $i++;
        }
        return $formattedData;
    }

    /**
     * @return array
     */
    public function setOptionsArgument(array $optionsArgument)
    {
        $this->optionsArgument = $optionsArgument;
    }

    /**
     * @return array
     */
    public function getOptionsArgument()
    {
        return $this->optionsArgument;
    }

    /**
     * @return array
     */
    public function getCsvHeader()
    {
        return $this->csvHeader;
    }

    /**
     * @return string
     */
    public function getBehavior()
    {
        return $this->behavior;
    }

    /**
     * @param string $behavior
     */
    public function setBehavior($behavior)
    {
        $this->behavior = $behavior;
    }

    /**
     * @return string
     */
    public function getEntityCode()
    {
        return $this->entityCode;
    }

    /**
     * @param string $entityCode
     */
    public function setEntityCode($entityCode)
    {
        $this->entityCode = $entityCode;
    }

    /**
     * @return string
     */
    public function getValidationStrategy()
    {
        return $this->validationStrategy;
    }

    /**
     * @param string $validationStrategy
     */
    public function setValidationStrategy($validationStrategy)
    {
        $this->validationStrategy = $validationStrategy;
    }

    /**
     * @return string
     */
    public function getAllowedErrorCount()
    {
        return $this->allowedErrorCount;
    }

    /**
     * @param string $allowedErrorCount
     */
    public function setAllowedErrorCount($allowedErrorCount)
    {
        $this->allowedErrorCount = $allowedErrorCount;
    }

    /**
     * @return string
     */
    public function getIgnoreDuplicates()
    {
        return $this->ignoreDuplicates;
    }

    /**
     * @param string $ignoreDuplicates
     */
    public function setIgnoreDuplicates($ignoreDuplicates)
    {
        $this->ignoreDuplicates = $ignoreDuplicates;
    }

    /**
     * @return string
     */
    public function getImportImagesFileDir()
    {
        return $this->importImagesFileDir;
    }

    /**
     * @param string $ignoreDuplicates
     */
    public function setImportImagesFileDir($importImagesFileDir = '')
    {
        $this->importImagesFileDir = $importImagesFileDir;
    }


    public function afterFinishImport(){}

    public function getLanguages() {
        $header = $this->getCsvHeader();
        $languages = [];
        if(count($header)) {
            foreach ($header as $key) {
                if (($currentStoreCode = $this->getStoreCodeFromFieldName($key)) && !in_array($currentStoreCode, $languages)) {
                    $languages[] = $currentStoreCode;
                }
            }
        }
        return $languages;
    }

    public function getStoreCodeFromFieldName($fieldName) {
        if(strrpos($fieldName, '_') !== false) {
            $codeString = substr($fieldName, strrpos($fieldName, '_') + 1);
            return ($this->getStoreExistByCode($codeString)) ? $codeString : false;
        }
    }

    public function getStoreExistByCode($storeCode) {
        $storeManager = $this->objectManager->get('Magento\Store\Model\StoreManagerInterface');
        $stores = $storeManager->getStores(true, false);
        foreach($stores as $store) {
            if($store->getCode() === $storeCode) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get Resource Connection
     *
     * @return \Magento\Framework\App\ResourceConnection
     */
    public function getConnectionPull(){
        return \Magento\Framework\App\ObjectManager::getInstance()
            ->get(\Magento\Framework\App\ResourceConnection::class);
    }

    public function clearCache(array $cacheTags) {
        echo PHP_EOL."cleaning cache...".PHP_EOL;
        try {
            $_cacheTypeList = $this->objectManager->create('Magento\Framework\App\Cache\TypeListInterface');
            $_cacheFrontendPool = $this->objectManager->create('Magento\Framework\App\Cache\Frontend\Pool');
            $types = ['config', 'layout', 'block_html', 'collections', 'reflection', 'db_ddl', 'eav', 'config_integration', 'config_integration_api', 'full_page', 'translate', 'config_webservice'];

            $cleanedCache = false;
            foreach ($cacheTags as $cacheTag) {
                if (in_array($cacheTag, $types)) {
                    $cleanedCache = true;
                    $_cacheTypeList->cleanType($cacheTag);
                }
            }

            if($cleanedCache) {
                foreach ($_cacheFrontendPool as $cacheFrontend) {
                    $cacheFrontend->getBackend()->clean();
                }
            }

        } catch (Exception $e) {
            echo $msg = 'Error : ' . $e->getMessage();
            die();
        }
    }

}
