<?php
/**
 * Downloadable
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Exception\AlreadyExistsException;

use Magento\Store\Model\StoreManager;
use Magento\Store\Model\Store;

class Urlrewrite extends Command
{
    /**
     * @var Magento\UrlRewrite\Model\UrlRewrite
     */
    private $urlRewrite;
    /**
     * @var Magento\UrlRewrite\Helper\UrlRewrite
     */
    private $urlRewriteHelper;
    /**
     * @var \Magento\Framework\File\Csv
     */
    private $csvProcessor;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    protected $objectManager;

    const FILE_PATH = 'pub/media/import/url_rewrite.csv';
    const STORE_CODE = 'it';

    /**
     * Urlrewrite constructor.
     * @param \Magento\UrlRewrite\Model\UrlRewrite $urlRewrite
     * @param \Magento\UrlRewrite\Helper\UrlRewrite $urlRewriteHelper
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     */
    public function __construct(
        \Magento\UrlRewrite\Model\UrlRewrite $urlRewrite,
        \Magento\UrlRewrite\Helper\UrlRewrite $urlRewriteHelper,
        \Magento\Framework\File\Csv $csvProcessor,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    )
    {
        $this->urlRewrite = $urlRewrite;
        $this->urlRewriteHelper = $urlRewriteHelper;
        $this->csvProcessor = $csvProcessor;
        $this->storeManager = $storeManager;
        parent::__construct();
    }

    protected function configure()
    {
        $this->setName('drop:import:url_rewrite')
            ->setDescription('Import Url Rewrite');
    }

    /**
     * Send last changed subscribers to Sendinblue
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(InputInterface $input, OutputInterface $output) {

        $data = $this->csvProcessor->getData(BP . '/' . self::FILE_PATH);
        $header = array_shift($data);
        $storeId = $this->getStoreIdByCode(self::STORE_CODE);

        $i = 0;
        $urlRewrites = [];
        foreach ($data as $row) {
            foreach ($row as $key => $value) {
                $urlRewrites[$i][$header[$key]] = $value;
            }
            $i++;
        }

        $successCounter = 0;
        $errorCounter = 0;
        $alreadyExistCounter = 0;
        $notFound = 0;
        foreach($urlRewrites as $key => $ur) {
            if((empty($ur['request_path'])) || (empty($ur['target_path']))) {
                echo PHP_EOL . "Empty request_path or target_path. Skipping";
                $notFound++;
                continue;
            }
            if(empty($ur['redirect_type'])) {
                $ur['redirect_type'] = 301;
            }

            $this->urlRewrite->load(0);

            $this->urlRewriteHelper->validateRequestPath($this->getFormattedRequestPath($ur['request_path']));
            $this->urlRewrite->setEntityType('custom')
                ->setRequestPath($this->getFormattedRequestPath($ur['request_path']))
                ->setTargetPath($this->getFormattedTargetPath($ur['target_path']))
                ->setRedirectType($ur['redirect_type'])
                ->setStoreId($storeId)
                ->setDescription("Imported Urls");

            try {
                $this->urlRewrite->save();
                unset($this->urlRewrite);
                $successCounter++;
            } catch (AlreadyExistsException $e) {
                echo PHP_EOL . "Url Rewrite already exist: " . $ur['request_path'] . ' - ' . $ur['target_path'] . ' - ' . $e->getMessage();
                $alreadyExistCounter++;
                continue;
            } catch (\Exception $e) {
                echo PHP_EOL . "FAILED to import Url Rewrite: " . $ur['request_path'] . ' - ' . $ur['target_path'] . ' - ' . $e->getMessage();
                $errorCounter++;
                continue;
            }
        }

        echo PHP_EOL;
        echo PHP_EOL . "Imported successfully " . $successCounter . ' url rewrite';
        echo PHP_EOL . "NOT IMPORTED " . $errorCounter . ' url rewrite';
        echo PHP_EOL . "Empty data for " . $notFound . ' url rewrite';
        echo PHP_EOL . "Already exist " . $alreadyExistCounter . ' url rewrite';
        echo PHP_EOL;
    }

    public function getStoreIdByCode($storeCode) {
        $stores = $this->storeManager->getStores(true, false);
        foreach($stores as $store){
            if($store->getCode() === $storeCode){
                return $store->getId();
            }
        }
        return false;
    }

    public function getFormattedRequestPath($url) {
        return $this->removeFirstCharacterIfSlash($this->getFormattedUrl($url));
    }

    public function getFormattedTargetPath($url) {
        return $this->removeFirstCharacterIfSlash($this->removeLanguageFromUrl($this->getFormattedUrl($url)));
    }

    public function getFormattedUrl($url) {
        return parse_url($url, PHP_URL_PATH);
    }

    public function removeLanguageFromUrl($url) {
        return str_replace('/'.self::STORE_CODE, '', $url);
    }

    public function removeFirstCharacterIfSlash($url) {
        return ltrim($url, '/');
    }

}
