<?php

namespace Drop\Import\Model\Import;

use Exception;
use Magento\CatalogImportExport\Model\Import\Product\RowValidatorInterface as ValidatorInterface;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Drop\Import\Logger\Logger;

class Attribute extends \Magento\ImportExport\Model\Import\Entity\AbstractEntity
{
    const COL_ATTRIBUTE_CODE = 'attribute_code';

    const ATTRIBUTE_FRONTEND_INPUT = 'frontend_input';
    const ATTRIBUTE_FRONTEND_LABEL = 'frontend_label';
    const ATTRIBUTE_OPTION_STORE = 'option_store';
    const ATTRIBUTE_OPTION_VALUE = 'option_value';

    /**
     * Validation failure message template definitions.
     *
     * @var array
     */
    protected $_messageTemplates = [
        'attributeCodeEmpty' => 'Attribute Code is empty',
        'attributeCodeNotFoundToDelete' => 'Attribute with specified Attribute Code not found',
        'attributeCodeAlreadyExist' => 'Attribute with specified Attribute Code already exist'
    ];

    /**
     * If we should check column names
     *
     * @var bool
     */
    protected $needColumnCheck = true;

    /**
     * Valid column names.
     *
     * @array
     */
    protected $validColumnNames = [
        self::COL_ATTRIBUTE_CODE,
        self::ATTRIBUTE_FRONTEND_INPUT,
        self::ATTRIBUTE_FRONTEND_LABEL,
        self::ATTRIBUTE_OPTION_STORE,
        self::ATTRIBUTE_OPTION_VALUE
    ];

    /**
     * Need to log in import history
     *
     * @var bool
     */
    protected $logInHistory = true;

    /**
     * Permanent entity columns.
     *
     * @var string[]
     */
    protected $_permanentAttributes = [self::COL_ATTRIBUTE_CODE];

    /**
     * ProductAttributeRepositoryInterface
     *
     * @var string
     */
    private $productAttributeRepository;

    /**
     * Eav
     *
     * @var string
     */
    private $eavAttribute;
    private $attributeOptionManagement;
    private $optionLabelFactory;
    private $optionFactory;
    private $storeManager;
    /**
     * @var Logger
     */
    private $logger;

    /**
     * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
     * @param \Magento\ImportExport\Model\ResourceModel\Import\Data $importData
     * @param \Magento\Framework\App\ResourceConnection $resource
     * @param ProcessingErrorAggregatorInterface $errorAggregator
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function __construct(
        //Mandatory dependencies
        \Magento\ImportExport\Model\ResourceModel\Import\Data $importData,
        ProcessingErrorAggregatorInterface $errorAggregator,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\ImportExport\Helper\Data $importExportData,
        \Magento\ImportExport\Model\ResourceModel\Helper $resourceHelper,
        \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory $resourceFactory,

        //Custom
        \Magento\Catalog\Model\ResourceModel\Eav\AttributeFactory $eavAttribute,
        \Magento\Catalog\Api\ProductAttributeRepositoryInterface\Proxy $productAttributeRepository,
        \Magento\Eav\Api\AttributeOptionManagementInterface $attributeOptionManagement,
        \Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory $optionLabelFactory,
        \Magento\Eav\Api\Data\AttributeOptionInterfaceFactory $optionFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute $attributeModel,
        \Drop\Import\Logger\Logger $logger
    )
    {
        $this->_dataSourceModel = $importData;
        $this->errorAggregator = $errorAggregator;
        $this->jsonHelper = $jsonHelper;
        $this->_resourceHelper = $resourceHelper;
        $this->_resourceFactory = $resourceFactory;
        $this->_importExportData = $importExportData;

        $this->productAttributeRepository = $productAttributeRepository;
        $this->attributeOptionManagement = $attributeOptionManagement;
        $this->optionLabelFactory = $optionLabelFactory;
        $this->optionFactory = $optionFactory;
        $this->storeManager = $storeManager;
        $this->eavAttribute = $eavAttribute;
        $this->logger = $logger;
        $this->attributeModel = $attributeModel;

        foreach (array_merge($this->errorMessageTemplates, $this->_messageTemplates) as $errorCode => $message) {
            $this->getErrorAggregator()->addErrorMessageTemplate($errorCode, $message);
        }
    }

    /**
     * Validator object getter.
     *
     * @param string $type
     */
    protected function _getValidator($type)
    {
        //TODO
        return true;
    }

    /**
     * Entity type code getter.
     *
     * @return string
     */
    public function getEntityTypeCode()
    {
        return 'catalog_attribute';
    }

    /**
     * Row validation.
     *
     * @param array $rowData
     * @param int $rowNum
     * @return bool
     */
    public function validateRow(array $rowData, $rowNum)
    {
        $attributeCode = false;
        if (isset($this->_validatedRows[$rowNum])) {
            return !$this->getErrorAggregator()->isRowInvalid($rowNum);
        }
        $this->_validatedRows[$rowNum] = true;
        // BEHAVIOR_DELETE use specific validation logic
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            if (!isset($rowData[self::COL_ATTRIBUTE_CODE])) {
                $this->addRowError('attributeCodeEmpty', $rowNum);
                return false;
            }
            return true;
        }
        //TODO
//        if (!$this->_getValidator(self::VALIDATOR_MAIN)->isValid($rowData)) {
//            foreach ($this->_getValidator(self::VALIDATOR_MAIN)->getMessages() as $message) {
//                $this->addRowError($message, $rowNum);
//            }
//        }
        if (isset($rowData[self::COL_ATTRIBUTE_CODE])) {
            $attributeCode = $rowData[self::COL_ATTRIBUTE_CODE];
        }
        if (false === $attributeCode) {
            $this->addRowError(ValidatorInterface::ERROR_ROW_IS_ORPHAN, $rowNum);
        }
        return !$this->getErrorAggregator()->isRowInvalid($rowNum);
    }

    /**
     * Create Attribute data from raw data.
     *
     * @throws Exception
     * @return bool Result of operation.
     */
    protected function _importData()
    {
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            $this->deleteAttribute();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $this->getBehavior()) {
            $this->saveAndReplaceAttribute();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $this->getBehavior()) {
            $this->saveAndReplaceAttribute();
        }
        return true;
    }

    /**
     * Deletes Attribute data from raw data.
     *
     * @return $this
     */
    public function deleteAttribute()
    {
        $listAttributes = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            foreach ($bunch as $rowNum => $rowData) {
                $this->validateRow($rowData, $rowNum);
                if (!$this->getErrorAggregator()->isRowInvalid($rowNum)) {
                    $rowAttribute = $rowData[self::COL_ATTRIBUTE_CODE];
                    $listAttributes[] = $rowAttribute;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                }
            }
        }
        if ($listAttributes) {
            $this->deleteAttributes(array_unique($listAttributes));
        }
        return $this;
    }

    /**
     * Save and replace Attribute
     *
     * @return $this
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    protected function saveAndReplaceAttribute()
    {
        $behavior = $this->getBehavior();
        $listAttributes = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            $attributes = [];
            $frontendLabel = [];
            $optionsData = [];

            foreach ($bunch as $rowNum => $rowData) {

                if (!$this->validateRow($rowData, $rowNum)) {
                    $this->addRowError('attributeCodeEmpty', $rowNum);
                    continue;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                    continue;
                }
                $rowAttribute = strtolower($rowData[self::COL_ATTRIBUTE_CODE]);
                $storeCode = $this->getStoreIdByCode($rowData[self::ATTRIBUTE_OPTION_STORE]);
                $listAttributes[] = $rowAttribute;

                if (!empty($rowData[self::ATTRIBUTE_OPTION_VALUE]) &&
                    in_array($rowData[self::ATTRIBUTE_FRONTEND_INPUT], ['select', 'multiselect'], true)
                ) {
                    $optionValues = explode('|', $rowData[self::ATTRIBUTE_OPTION_VALUE]);
                    $optionsCounter = 0;
                    foreach ($optionValues as $optionValue) {
                        if(!empty($optionValue)) {
                            $optionsData[$rowAttribute]['option_' . $optionsCounter][$storeCode] = $optionValue;
                            $optionsCounter++;
                        }
                    }
                }

                $attribute_id = $this->attributeModel->getIdByCode("catalog_product",$rowAttribute);

                $frontendLabel[$storeCode] = $rowData[self::ATTRIBUTE_FRONTEND_LABEL];

                $attributes[$rowAttribute] = [
                    'attribute_id' => $attribute_id,
                    'entity_type_id' => 4, //TODO: Get from dependencies
                    'attribute_code' => $rowAttribute,
                    'backend_type' => 'int',
                    'frontend_input' => $rowData[self::ATTRIBUTE_FRONTEND_INPUT],
                    'frontend_label' => $frontendLabel,
                    'is_global' => 1,
                    'is_visible' => 1,
                    'is_required' => 0,
                    'is_user_defined' => 1,
                    'is_searchable' => 0,
                    'is_filterable' => 0,
                    'is_comparable' => 0,
                    'is_visible_on_front' => 0,
                    'used_in_product_listing' => 0,
                    'is_html_allowed_on_front' => 1,
                    'is_unique' => 0,
                    'is_used_for_price_rules' => 0,
                    'is_filterable_in_search' => 0,
                    'used_in_product_listing' => 0,
                    'used_for_sort_by' => 0,
                    'is_visible_in_advanced_search' => 1,
                    'is_wysiwyg_enabled' => 0,
                    'is_used_for_promo_rules' => 0,
                    'is_required_in_admin_store' => 0,
                    'is_used_in_grid' => 0,
                    'is_visible_in_grid' => 0,
                    'is_filterable_in_grid' => 0
                ];


                if (!empty($optionsData) && !empty($optionsData[$rowAttribute])) {
                    $attributes[$rowAttribute]['option']['value'] = $optionsData[$rowAttribute];
                }
            }

            if (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $behavior) {
                if ($listAttributes) {
                    if ($this->deleteAttributes(array_unique($listAttributes))) {
                        $this->saveAttributes($attributes);
                    }
                }
            } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $behavior) {
                $this->saveAttributes($attributes);
            }

        }

        return $this;
    }

    /**
     * Save attributes
     *
     * @param array $attributeDatas
     */
    public function saveAttributes(array $attributeDatas)
    {
        if ($attributeDatas) {
            foreach ($attributeDatas as $attributeCode => $attributeData) {
                $attribute = $this->eavAttribute->create();
                $attribute->setData($attributeData);

                try {
                    $this->productAttributeRepository->save($attribute);
                } catch (\Magento\Framework\Exception\AlreadyExistsException $ex) {
                    $this->logger->info($ex->getMessage() . ': ' . $attributeCode);
                    //$this->addRowError($ex->getMessage().' - attributeCodeAlreadyExist: '.$attributeCode, 0);
                } catch (\Exception $ex) {
                    $this->logger->info($ex->getMessage() . ': ' . $attributeCode);
                    $this->addRowError($ex->getMessage(). ' - attributeCodeException: '.$attributeCode, 0);
                }
            }
        }
    }

    /**
     * Deletes attributes.
     *
     * @param array $attributeCodes
     */
    public function deleteAttributes(array $attributeCodes)
    {
        foreach ($attributeCodes as $attributeCode) {
            try {
                $this->productAttributeRepository->deleteById($attributeCode);
            } catch (\Exception $ex) {
                $this->logger->info($ex->getMessage());
                $this->addRowError($ex->getMessage());
            }
        }
    }

    /**
     * Get Store Id By Code
     * @param type $storeCode
     * @return int
     */
    public function getStoreIdByCode($storeCode)
    {
        $stores = $this->storeManager->getStores(true, false);
        foreach ($stores as $store) {
            if ($store->getCode() === $storeCode) {
                return $store->getId();
            }
        }
        return 0;
    }

}
