<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Import;

use Magento\CatalogImportExport\Model\Import\Product\RowValidatorInterface as ValidatorInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingError;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Drop\Import\Logger\Logger;

/**
 * Class AttributeSet
 * @package Drop\Import\Model\Import
 */
class AttributeSet extends \Magento\ImportExport\Model\Import\Entity\AbstractEntity
{
    const COL_ATTRIBUTE_SET_CODE = 'attribute_set_name';
    const COL_ATTRIBUTE_CODE = 'attribute_code';
    const ATTRIBUTE_FRONTEND_LABEL = 'frontend_label';
    const ATTRIBUTE_OPTION_STORE = 'option_store';
    const ATTRIBUTE_OPTION_VALUE = 'option_value';

    /**
     * Validation failure message template definitions.
     *
     * @var array
     */
    protected $_messageTemplates = [
        'attributeSetNameEmpty'                  => 'Attribute Set Code is empty',
        'attributeSetNameNotFoundToDelete'       => 'Attribute Set with specified Attribute Code not found',
        'attributeSetNameAlreadyExist' => 'Attribute Set with specified Attribute Code already exist',
        'attributeCodeAlreadyExist' => 'Attribute with specified Attribute Code already exist and all the options are correctly configured'
    ];


    /**
     * If we should check column names
     *
     * @var bool
     */
    protected $needColumnCheck = true;

    /**
     * Valid column names.
     *
     * @array
     */
    protected $validColumnNames = [
        self::COL_ATTRIBUTE_SET_CODE,
        self::COL_ATTRIBUTE_CODE,
        self::ATTRIBUTE_FRONTEND_LABEL,
        self::ATTRIBUTE_OPTION_STORE,
        self::ATTRIBUTE_OPTION_VALUE,
    ];

    /**
     * Need to log in import history
     *
     * @var bool
     */
    protected $logInHistory = true;

    /**
     * Permanent entity columns.
     *
     * @var string[]
     */
    protected $_permanentAttributes = [self::COL_ATTRIBUTE_SET_CODE];
    /**
     * @var \Magento\Eav\Model\AttributeSetManagement
     */
    private $attributeSetManagement;
    /**
     * @var \Magento\Eav\Model\AttributeSetRepository
     */
    private $attributeSetRepositoryFactory;
    /**
     * @var \Magento\Eav\Model\Entity\Type
     */
    private $entityTypeFactory;
    /**
     * @var \Magento\Eav\Model\Entity\Attribute\Set
     */
    private $entityAttributeSetFactory;
    /**
     * @var \Magento\Eav\Model\AttributeManagement
     */
    private $attributeManagement;
    /**
     * @var \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory
     */
    private $_resourceFactory;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var Attribute
     */
    private $importAttribute;
    /**
     * @var Logger
     */
    private $logger;
    /**
     * Json Serializer Instance
     *
     * @var Json
     */
    private $serializer;
    /**
     * @var \Magento\Eav\Model\ResourceModel\Entity\Attribute
     */
    private $attributeModel;

    /**
     * AttributeSet constructor.
     * @param \Magento\ImportExport\Model\ResourceModel\Import\Data $importData
     * @param ProcessingErrorAggregatorInterface $errorAggregator
     * @param \Magento\Framework\Json\Helper\Data $jsonHelper
     * @param \Magento\ImportExport\Helper\Data $importExportData
     * @param \Magento\ImportExport\Model\ResourceModel\Helper $resourceHelper
     * @param \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory $resourceFactory
     * @param \Magento\Eav\Model\AttributeSetManagement $attributeSetManagement
     * @param \Magento\Eav\Model\AttributeManagement $attributeManagement
     * @param \Magento\Eav\Model\AttributeSetRepositoryFactory $attributeSetRepositoryFactory
     * @param \Magento\Eav\Model\Entity\TypeFactory $entityTypeFactory
     * @param \Magento\Eav\Model\Entity\Attribute\SetFactory $entityAttributeSetFactory
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param Attribute $importAttribute
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute $attributeModel
     * @param \Drop\Import\Logger\Logger $logger
     */
    public function __construct(
        \Magento\ImportExport\Model\ResourceModel\Import\Data $importData,
        ProcessingErrorAggregatorInterface $errorAggregator,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\ImportExport\Helper\Data $importExportData,
        \Magento\ImportExport\Model\ResourceModel\Helper $resourceHelper,
        \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory $resourceFactory,
        \Magento\Eav\Model\AttributeSetManagement $attributeSetManagement,
        \Magento\Eav\Model\AttributeManagement $attributeManagement,
        \Magento\Eav\Model\AttributeSetRepositoryFactory $attributeSetRepositoryFactory,
        \Magento\Eav\Model\Entity\TypeFactory $entityTypeFactory,
        \Magento\Eav\Model\Entity\Attribute\SetFactory $entityAttributeSetFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Drop\Import\Model\Import\Attribute $importAttribute,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute $attributeModel,
        \Drop\Import\Logger\Logger $logger
    )
    {
        $this->_dataSourceModel = $importData;
        $this->errorAggregator = $errorAggregator;
        $this->jsonHelper = $jsonHelper;
        $this->_resourceHelper = $resourceHelper;
        $this->_resourceFactory = $resourceFactory;
        $this->_importExportData = $importExportData;

        $this->attributeSetManagement = $attributeSetManagement;
        $this->attributeSetRepositoryFactory = $attributeSetRepositoryFactory;
        $this->entityTypeFactory = $entityTypeFactory;
        $this->entityAttributeSetFactory = $entityAttributeSetFactory;
        $this->attributeManagement = $attributeManagement;
        $this->storeManager = $storeManager;
        $this->importAttribute = $importAttribute;
        $this->logger = $logger;
        $this->attributeModel = $attributeModel;
    }

    /**
     * Validator object getter.
     *
     * @param string $type
     */
    protected function _getValidator($type)
    {
        //TODO
        return true;
    }

    /**
     * Entity type code getter.
     *
     * @return string
     */
    public function getEntityTypeCode()
    {
        return 'catalog_attributeset';
    }

    /**
     * Row validation.
     *
     * @param array $rowData
     * @param int $rowNum
     * @return bool
     */
    public function validateRow(array $rowData, $rowNum)
    {
        $attributeSetName = false;
        if (isset($this->_validatedRows[$rowNum])) {
            return !$this->getErrorAggregator()->isRowInvalid($rowNum);
        }
        $this->_validatedRows[$rowNum] = true;
        // BEHAVIOR_DELETE use specific validation logic
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            if (!isset($rowData[self::COL_ATTRIBUTE_SET_CODE])) {
                $this->addRowError('attributeSetNameEmpty', $rowNum);
                return false;
            }
            return true;
        }
        //TODO
//        if (!$this->_getValidator(self::VALIDATOR_MAIN)->isValid($rowData)) {
//            foreach ($this->_getValidator(self::VALIDATOR_MAIN)->getMessages() as $message) {
//                $this->addRowError($message, $rowNum);
//            }
//        }
        if (isset($rowData[self::COL_ATTRIBUTE_SET_CODE])) {
            $attributeSetName = $rowData[self::COL_ATTRIBUTE_SET_CODE];
        }
        if (false === $attributeSetName) {
            $this->addRowError(ValidatorInterface::ERROR_ROW_IS_ORPHAN, $rowNum);
        }
        return !$this->getErrorAggregator()->isRowInvalid($rowNum);
    }

    /**
     * Create Attribute data from raw data.
     *
     * @throws \Exception
     * @return bool Result of operation.
     */
    protected function _importData()
    {
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            $this->deleteAttributeSet();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $this->getBehavior()) {
            $this->saveAndReplaceAttributeSet();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $this->getBehavior()) {
            $this->saveAndReplaceAttributeSet();
        }
        return true;
    }

    /**
     * Deletes Attribute data from raw data.
     *
     * @return $this
     */
    public function deleteAttributeSet()
    {
        $listAttributes = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            foreach ($bunch as $rowNum => $rowData) {
                $this->validateRow($rowData, $rowNum);
                if (!$this->getErrorAggregator()->isRowInvalid($rowNum)) {
                    $rowAttribute = $rowData[self::COL_ATTRIBUTE_SET_CODE];
                    $listAttributes[] = $rowAttribute;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                }
            }
        }
        if ($listAttributes) {
            $this->deleteAttributeSets(array_unique($listAttributes));
        }
        return $this;
    }

    /**
     * Save and replace Attribute
     *
     * @return $this
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    protected function saveAndReplaceAttributeSet()
    {
        $behavior = $this->getBehavior();
        $attributeSets = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            $attributeSets = [];
            $attributes = [];
            $optionValues = [];
            $optionsData = [];

            foreach ($bunch as $rowNum => $rowData) {
                if (!$this->validateRow($rowData, $rowNum)) {
                    $this->addRowError('attributeSetNameEmpty', $rowNum);
                    continue;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                    continue;
                }

                $storeCode = $this->importAttribute->getStoreIdByCode($rowData[self::ATTRIBUTE_OPTION_STORE]);
                $attributeSetName = $rowData[self::COL_ATTRIBUTE_SET_CODE];
                $attributeCode = strtolower($rowData[self::COL_ATTRIBUTE_CODE]);
                if (empty($attributeCode) || empty($rowData[self::ATTRIBUTE_OPTION_VALUE])) {
                    continue;
                }

                $attribute_id = $this->attributeModel->getIdByCode("catalog_product", $attributeCode);
                $optionValues = explode('|', $rowData[self::ATTRIBUTE_OPTION_VALUE]);
                if(!empty($attribute_id)) {
                    $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
                    $alreadyExistAttribute = $objectManager->get(\Magento\Eav\Model\Entity\Attribute::class)->load($attribute_id);
                    $alreadyExistOptions = $alreadyExistAttribute->getSource()->getAllOptions();
                    $alreadyExistLabel = [];
                    foreach($alreadyExistOptions as $alreadyExistOption) {
                        $alreadyExistLabel[] = $alreadyExistOption['label'];
                    }

                    $attributeExistCounter = 0;
                    foreach($optionValues as $optionValue) {
                        if(in_array($optionValue, $alreadyExistLabel)) {
                            unset($optionValues[$attributeExistCounter]);
                        }
                        $attributeExistCounter++;
                    }

                    if(!count($optionValues)) {
                        $attributeSets[$attributeSetName] = $attributeCode;
                        $this->addRowError('attributeCodeAlreadyExist', $this->_processedRowsCount, null, null, ProcessingError::ERROR_LEVEL_NOT_CRITICAL);
                        if(!empty($storeCode)) {
                            $this->_processedRowsCount++;
                        }
                        continue;
                    }
                }

                $optionsCounter = 0;
                foreach ($optionValues as $optionValue) {
                    if (!empty($optionValue)) {
                        $optionsData[$attributeCode]['option_' . $optionsCounter][$storeCode] = $optionValue;
                        $optionsCounter++;
                    }
                }

                $frontendLabel[$storeCode] = $rowData[self::ATTRIBUTE_FRONTEND_LABEL];
                $attributes[$attributeCode] = [
                    'attribute_id' => $attribute_id,
                    'entity_type_id'                => 4, //TODO: Get from dependencies
                    'attribute_code'                => $attributeCode,
                    'backend_type'                  => 'int',
                    'frontend_input'                => 'select',
                    'frontend_label'                => $frontendLabel,
                    'is_global'                     => 1,
                    'is_visible'                    => 1,
                    'is_required'                   => 0,
                    'is_user_defined'               => 1,
                    'is_searchable'                 => 0,
                    'is_filterable'                 => 0,
                    'is_comparable'                 => 0,
                    'is_visible_on_front'           => 0,
                    'used_in_product_listing'       => 0,
                    'is_html_allowed_on_front'      => 1,
                    'is_unique'                     => 0,
                    'is_used_for_price_rules'       => 0,
                    'is_filterable_in_search'       => 0,
                    'used_in_product_listing'       => 0,
                    'used_for_sort_by'              => 0,
                    'is_visible_in_advanced_search' => 1,
                    'is_wysiwyg_enabled'            => 0,
                    'is_used_for_promo_rules'       => 0,
                    'is_required_in_admin_store'    => 0,
                    'is_used_in_grid'               => 0,
                    'is_visible_in_grid'            => 0,
                    'is_filterable_in_grid'         => 0,
                    'note'                          => 'super_attribute'
                ];
                if (!empty($optionsData)) {
                    $attributes[$attributeCode]['option']['value'] = $optionsData[$attributeCode];
                }

                $attributeSets[$attributeSetName] = $attributeCode;
            }

            if (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $behavior) {
                if ($attributeSets) {
                    if ($this->deleteAttributeSets(array_unique($attributeSets))) {
                        $this->importAttribute->saveAttributes($attributes);
                        $this->saveAttributeSets($attributeSets);
                    }
                }
            } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $behavior) {
                $this->importAttribute->saveAttributes($attributes);
                $this->saveAttributeSets($attributeSets);
            }
        }

        return $this;
    }

    /**
     * Save attributes
     *
     * @param array $attributeSetDatas
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function saveAttributeSets(array $attributeSetDatas)
    {

        if ($attributeSetDatas) {
            $entityTypeCode = 'catalog_product';
            /** @var \Magento\Eav\Model\Entity\Type $entityType */
            $entityType = $this->entityTypeFactory->create()->loadByCode($entityTypeCode);
            $defaultSetId = $entityType->getDefaultAttributeSetId();
            $rowCounter = 0;
            foreach ($attributeSetDatas as $attributeSetName => $attributeCode) {

                $data = [
                    'attribute_set_name' => $attributeSetName,
                    'entity_type_id'     => $entityType->getId(),
                    'sort_order'         => 100
                ];

                try {
                    $attributeSet = $this->entityAttributeSetFactory->create();
                    $attributeSet->setData($data);

                    $this->attributeSetManagement->create($entityTypeCode, $attributeSet, $defaultSetId);

                    if (!empty($attributeCode)) {
                        $this->attributeManagement->assign(
                            $entityTypeCode,
                            $attributeSet->getId(),
                            $attributeSet->getDefaultGroupId(), //TODO: Add possibility to customize Group
                            $attributeCode,
                            $attributeSet->getCollection()->count() * 10
                        );
                    }
                } catch (\Exception $e) {
                    $this->logger->info($e->getMessage() . ': ' . $attributeSetName);
                    $this->addRowError($e->getMessage() . ' ' . $attributeSetName, $rowCounter);
                }

                $rowCounter++;
            }
        }
    }

    protected function _saveValidatedBunches()
    {
        $source = $this->_getSource();
        $currentDataSize = 0;
        $bunchRows = [];
        $startNewBunch = false;
        $nextRowBackup = [];
        $maxDataSize = $this->_resourceHelper->getMaxDataSize();
        $bunchSize = $this->_importExportData->getBunchSize();
        $skuSet = [];

        $source->rewind();
        $this->_dataSourceModel->cleanBunches();

        while ($source->valid() || $bunchRows) {
            if ($startNewBunch || !$source->valid()) {
                $this->_dataSourceModel->saveBunch($this->getEntityTypeCode(), $this->getBehavior(), $bunchRows);

                $bunchRows = $nextRowBackup;
                $currentDataSize = strlen($this->getSerializer()->serialize($bunchRows));
                $startNewBunch = false;
                $nextRowBackup = [];
            }
            if ($source->valid()) {
                try {
                    $rowData = $source->current();
                    $skuSet[$rowData[self::COL_ATTRIBUTE_SET_CODE]] = true;
                } catch (\InvalidArgumentException $e) {
                    $this->addRowError($e->getMessage(), $this->_processedRowsCount);
                    $this->_processedRowsCount++;
                    $source->next();
                    continue;
                }


                $this->_processedRowsCount++;

                if ($this->validateRow($rowData, $source->key())) {
                    // add row to bunch for save
                    $rowData = $this->_prepareRowForDb($rowData);
                    $rowSize = strlen($this->jsonHelper->jsonEncode($rowData));

                    $isBunchSizeExceeded = $bunchSize > 0 && count($bunchRows) >= $bunchSize;

                    if ($currentDataSize + $rowSize >= $maxDataSize || $isBunchSizeExceeded) {
                        $startNewBunch = true;
                        $nextRowBackup = [$source->key() => $rowData];
                    } else {
                        $bunchRows[$source->key()] = $rowData;
                        $currentDataSize += $rowSize;
                    }
                }
                $source->next();
            }
        }
        $this->_processedEntitiesCount = count($skuSet);

        return $this;
    }

    private function getSerializer()
    {
        if (null === $this->serializer) {
            $this->serializer = ObjectManager::getInstance()->get(Json::class);
        }
        return $this->serializer;
    }

    /**
     * Deletes attributes.
     *
     * @param array $attributeSetNames
     */
    protected function deleteAttributeSets(array $attributeSetNames)
    {
        /**
         * TODO: Delete Attribute Set
         */
        die(__METHOD__ . '1');
        /*foreach ($attributeSetNames as $attributeSetName) {
            try {
                $this->productAttributeRepository->deleteById($attributeSetName);
            } catch (\Exception $ex) {
                $this->logger->info($ex->getMessage());
                $this->addRowError($ex->getMessage());
            }
        }*/
    }
}
