<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Import;

use Exception;
use Magento\CatalogImportExport\Model\Import\Product\RowValidatorInterface as ValidatorInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingError;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Drop\Import\Logger\Logger;

class Attribute extends \Magento\ImportExport\Model\Import\Entity\AbstractEntity
{
    const COL_ATTRIBUTE_CODE = 'attribute_code';

    const ATTRIBUTE_FRONTEND_INPUT = 'frontend_input';
    const ATTRIBUTE_FRONTEND_LABEL = 'frontend_label';
    const ATTRIBUTE_SCOPE = 'scope';
    const ATTRIBUTE_OPTION_STORE = 'option_store';
    const ATTRIBUTE_OPTION_VALUE = 'option_value';

    /**
     * Validation failure message template definitions.
     *
     * @var array
     */
    protected $_messageTemplates = [
        'attributeCodeEmpty' => 'Attribute Code is empty',
        'attributeCodeNotFoundToDelete' => 'Attribute with specified Attribute Code not found',
        'attributeCodeAlreadyExist' => 'Attribute with specified Attribute Code already exist and all the options are correctly configured'
    ];

    /**
     * If we should check column names
     *
     * @var bool
     */
    protected $needColumnCheck = true;

    /**
     * Valid column names.
     *
     * @array
     */
    protected $validColumnNames = [
        self::COL_ATTRIBUTE_CODE,
        self::ATTRIBUTE_FRONTEND_INPUT,
        self::ATTRIBUTE_FRONTEND_LABEL,
        self::ATTRIBUTE_SCOPE,
        self::ATTRIBUTE_OPTION_STORE,
        self::ATTRIBUTE_OPTION_VALUE
    ];

    /**
     * Need to log in import history
     *
     * @var bool
     */
    protected $logInHistory = true;

    /**
     * Permanent entity columns.
     *
     * @var string[]
     */
    protected $_permanentAttributes = [self::COL_ATTRIBUTE_CODE];

    /**
     * @var \Magento\Catalog\Api\ProductAttributeRepositoryInterface\Proxy
     */
    private $productAttributeRepository;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Eav\AttributeFactory
     */
    private $eavAttribute;
    private $attributeOptionManagement;
    /**
     * @var \Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory
     */
    private $optionLabelFactory;
    /**
     * @var \Magento\Eav\Api\Data\AttributeOptionInterfaceFactory
     */
    private $optionFactory;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var Logger
     */
    private $logger;
    /**
     * Json Serializer Instance
     *
     * @var Json
     */
    private $serializer;
    /**
     * @var \Magento\Eav\Model\ResourceModel\Entity\Attribute
     */
    private $attributeModel;
    /**
     * @var \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory
     */
    private $_resourceFactory;
    /**
     * @var \Magento\Eav\Model\AttributeManagement
     */
    private $attributeManagement;
    /**
     * @var \Magento\Catalog\Model\Product\AttributeSet\OptionsFactory
     */
    private $optionsFactory;
    /**
     * @var \Magento\Eav\Api\AttributeSetRepositoryInterface
     */
    private $attributeSet;

    /**
     * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
     * @param \Magento\ImportExport\Model\ResourceModel\Import\Data $importData
     * @param ProcessingErrorAggregatorInterface $errorAggregator
     * @param \Magento\Framework\Json\Helper\Data $jsonHelper
     * @param \Magento\ImportExport\Helper\Data $importExportData
     * @param \Magento\ImportExport\Model\ResourceModel\Helper $resourceHelper
     * @param \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory $resourceFactory
     * @param \Magento\Catalog\Model\ResourceModel\Eav\AttributeFactory $eavAttribute
     * @param \Magento\Catalog\Api\ProductAttributeRepositoryInterface\Proxy $productAttributeRepository
     * @param \Magento\Eav\Api\AttributeOptionManagementInterface $attributeOptionManagement
     * @param \Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory $optionLabelFactory
     * @param \Magento\Eav\Api\Data\AttributeOptionInterfaceFactory $optionFactory
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute $attributeModel
     * @param \Magento\Eav\Model\AttributeManagement $attributeManagement
     * @param \Magento\Catalog\Model\Product\AttributeSet\OptionsFactory $optionsFactory
     * @param \Magento\Eav\Api\AttributeSetRepositoryInterface $attributeSet
     * @param \Drop\Import\Logger\Logger $logger
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function __construct(
        //Mandatory dependencies
        \Magento\ImportExport\Model\ResourceModel\Import\Data $importData,
        ProcessingErrorAggregatorInterface $errorAggregator,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\ImportExport\Helper\Data $importExportData,
        \Magento\ImportExport\Model\ResourceModel\Helper $resourceHelper,
        \Magento\CatalogImportExport\Model\Import\Proxy\Product\ResourceModelFactory $resourceFactory,

        //Custom
        \Magento\Catalog\Model\ResourceModel\Eav\AttributeFactory $eavAttribute,
        \Magento\Catalog\Api\ProductAttributeRepositoryInterface\Proxy $productAttributeRepository,
        \Magento\Eav\Api\AttributeOptionManagementInterface $attributeOptionManagement,
        \Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory $optionLabelFactory,
        \Magento\Eav\Api\Data\AttributeOptionInterfaceFactory $optionFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute $attributeModel,
        \Magento\Eav\Model\AttributeManagement $attributeManagement,
        \Magento\Catalog\Model\Product\AttributeSet\OptionsFactory $optionsFactory,
        \Magento\Eav\Api\AttributeSetRepositoryInterface $attributeSet,
        \Drop\Import\Logger\Logger $logger
    )
    {
        $this->_dataSourceModel = $importData;
        $this->errorAggregator = $errorAggregator;
        $this->jsonHelper = $jsonHelper;
        $this->_resourceHelper = $resourceHelper;
        $this->_resourceFactory = $resourceFactory;
        $this->_importExportData = $importExportData;

        $this->productAttributeRepository = $productAttributeRepository;
        $this->attributeOptionManagement = $attributeOptionManagement;
        $this->optionLabelFactory = $optionLabelFactory;
        $this->optionFactory = $optionFactory;
        $this->storeManager = $storeManager;
        $this->eavAttribute = $eavAttribute;
        $this->attributeModel = $attributeModel;
        $this->attributeManagement = $attributeManagement;
        $this->optionsFactory = $optionsFactory;
        $this->attributeSet = $attributeSet;
        $this->logger = $logger;

        foreach (array_merge($this->errorMessageTemplates, $this->_messageTemplates) as $errorCode => $message) {
            $this->getErrorAggregator()->addErrorMessageTemplate($errorCode, $message);
        }
    }

    /**
     * Validator object getter.
     *
     * @param string $type
     */
    protected function _getValidator($type)
    {
        //TODO
        return true;
    }

    /**
     * Entity type code getter.
     *
     * @return string
     */
    public function getEntityTypeCode()
    {
        return 'catalog_attribute';
    }

    protected function _saveValidatedBunches()
    {
        $source = $this->_getSource();
        $currentDataSize = 0;
        $bunchRows = [];
        $startNewBunch = false;
        $nextRowBackup = [];
        $maxDataSize = $this->_resourceHelper->getMaxDataSize();
        $bunchSize = $this->_importExportData->getBunchSize();
        $skuSet = [];

        $source->rewind();
        $this->_dataSourceModel->cleanBunches();

        while ($source->valid() || $bunchRows) {
            if ($startNewBunch || !$source->valid()) {
                $this->_dataSourceModel->saveBunch($this->getEntityTypeCode(), $this->getBehavior(), $bunchRows);

                $bunchRows = $nextRowBackup;
                $currentDataSize = strlen($this->getSerializer()->serialize($bunchRows));
                $startNewBunch = false;
                $nextRowBackup = [];
            }
            if ($source->valid()) {
                try {
                    $rowData = $source->current();
                    $skuSet[$rowData[self::COL_ATTRIBUTE_CODE]] = true;
                } catch (\InvalidArgumentException $e) {
                    $this->addRowError($e->getMessage(), $this->_processedRowsCount);
                    $this->_processedRowsCount++;
                    $source->next();
                    continue;
                }


                $this->_processedRowsCount++;

                if ($this->validateRow($rowData, $source->key())) {
                    // add row to bunch for save
                    $rowData = $this->_prepareRowForDb($rowData);
                    $rowSize = strlen($this->jsonHelper->jsonEncode($rowData));

                    $isBunchSizeExceeded = $bunchSize > 0 && count($bunchRows) >= $bunchSize;

                    if ($currentDataSize + $rowSize >= $maxDataSize || $isBunchSizeExceeded) {
                        $startNewBunch = true;
                        $nextRowBackup = [$source->key() => $rowData];
                    } else {
                        $bunchRows[$source->key()] = $rowData;
                        $currentDataSize += $rowSize;
                    }
                }
                $source->next();
            }
        }
        $this->_processedEntitiesCount = count($skuSet);

        return $this;
    }

    private function getSerializer()
    {
        if (null === $this->serializer) {
            $this->serializer = ObjectManager::getInstance()->get(Json::class);
        }
        return $this->serializer;
    }

    /**
     * Row validation.
     *
     * @param array $rowData
     * @param int $rowNum
     * @return bool
     */
    public function validateRow(array $rowData, $rowNum)
    {
        $attributeCode = false;
        if (isset($this->_validatedRows[$rowNum])) {
            return !$this->getErrorAggregator()->isRowInvalid($rowNum);
        }
        $this->_validatedRows[$rowNum] = true;
        // BEHAVIOR_DELETE use specific validation logic
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            if (!isset($rowData[self::COL_ATTRIBUTE_CODE])) {
                $this->addRowError('attributeCodeEmpty', $rowNum);
                return false;
            }
            return true;
        }
        //TODO
//        if (!$this->_getValidator(self::VALIDATOR_MAIN)->isValid($rowData)) {
//            foreach ($this->_getValidator(self::VALIDATOR_MAIN)->getMessages() as $message) {
//                $this->addRowError($message, $rowNum);
//            }
//        }
        if (isset($rowData[self::COL_ATTRIBUTE_CODE])) {
            $attributeCode = $rowData[self::COL_ATTRIBUTE_CODE];
        }
        if (false === $attributeCode) {
            $this->addRowError(ValidatorInterface::ERROR_ROW_IS_ORPHAN, $rowNum);
        }
        return !$this->getErrorAggregator()->isRowInvalid($rowNum);
    }

    /**
     * Create Attribute data from raw data.
     *
     * @throws Exception
     * @return bool Result of operation.
     */
    protected function _importData()
    {
        if (\Magento\ImportExport\Model\Import::BEHAVIOR_DELETE == $this->getBehavior()) {
            $this->deleteAttribute();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $this->getBehavior()) {
            $this->saveAndReplaceAttribute();
        } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $this->getBehavior()) {
            $this->saveAndReplaceAttribute();
        }
        return true;
    }

    /**
     * Deletes Attribute data from raw data.
     *
     * @return $this
     */
    public function deleteAttribute()
    {
        $listAttributes = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            foreach ($bunch as $rowNum => $rowData) {
                $this->validateRow($rowData, $rowNum);
                if (!$this->getErrorAggregator()->isRowInvalid($rowNum)) {
                    $rowAttribute = $rowData[self::COL_ATTRIBUTE_CODE];
                    $listAttributes[] = $rowAttribute;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                }
            }
        }
        if ($listAttributes) {
            $this->deleteAttributes(array_unique($listAttributes));
        }
        return $this;
    }

    /**
     * Save and replace Attribute
     *
     * @return $this
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    protected function saveAndReplaceAttribute()
    {
        $behavior = $this->getBehavior();
        $listAttributes = [];
        while ($bunch = $this->_dataSourceModel->getNextBunch()) {
            $attributes = [];
            $frontendLabel = [];
            $optionsData = [];

            foreach ($bunch as $rowNum => $rowData) {

                if (!$this->validateRow($rowData, $rowNum)) {
                    $this->addRowError('attributeCodeEmpty', $rowNum);
                    continue;
                }
                if ($this->getErrorAggregator()->hasToBeTerminated()) {
                    $this->getErrorAggregator()->addRowToSkip($rowNum);
                    continue;
                }
                $rowAttribute = strtolower($rowData[self::COL_ATTRIBUTE_CODE]);
                $storeCode = $this->getStoreIdByCode($rowData[self::ATTRIBUTE_OPTION_STORE]);
                $listAttributes[] = $rowAttribute;

                $attribute_id = $this->attributeModel->getIdByCode("catalog_product", $rowAttribute);

                if (!empty($rowData[self::ATTRIBUTE_OPTION_VALUE]) &&
                    in_array($rowData[self::ATTRIBUTE_FRONTEND_INPUT], ['select', 'multiselect'], true)
                ) {
                    $optionValues = explode('|', $rowData[self::ATTRIBUTE_OPTION_VALUE]);

                    if(!empty($attribute_id)) {
                        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
                        $alreadyExistAttribute = $objectManager->get(\Magento\Eav\Model\Entity\Attribute::class)->load($attribute_id);
                        $alreadyExistOptions = $alreadyExistAttribute->getSource()->getAllOptions();
                        $alreadyExistLabel = [];
                        foreach($alreadyExistOptions as $alreadyExistOption) {
                            $alreadyExistLabel[] = $alreadyExistOption['label'];
                        }

                        $attributeExistCounter = 0;
                        foreach($optionValues as $optionValue) {
                            if(in_array($optionValue, $alreadyExistLabel)) {
                                unset($optionValues[$attributeExistCounter]);
                            }
                            $attributeExistCounter++;
                        }

                        if(!count($optionValues)) {
                            $this->addRowError('attributeCodeAlreadyExist', $this->_processedRowsCount, null, null, ProcessingError::ERROR_LEVEL_NOT_CRITICAL);
                            $this->_processedRowsCount++;
                            return $this;
                        }
                    }

                    $optionsCounter = 0;
                    foreach ($optionValues as $optionValue) {
                        if(!empty($optionValue)) {
                            $optionsData[$rowAttribute]['option_' . $optionsCounter][$storeCode] = $optionValue;
                            $optionsCounter++;
                        }
                    }
                }

                $frontendLabel[$storeCode] = $rowData[self::ATTRIBUTE_FRONTEND_LABEL];
                $isGlobal = ($rowData[self::ATTRIBUTE_SCOPE] == 'global') ? 1 : 0;
                $attributes[$rowAttribute] = [
                    'attribute_id' => $attribute_id,
                    'entity_type_id' => 4, //TODO: Get from dependencies
                    'attribute_code' => $rowAttribute,
                    'backend_type' => 'int',
                    'frontend_input' => $rowData[self::ATTRIBUTE_FRONTEND_INPUT],
                    'frontend_label' => $frontendLabel,
                    'is_global' => $isGlobal,
                    'is_visible' => 1,
                    'is_required' => 0,
                    'is_user_defined' => 1,
                    'is_searchable' => 0,
                    'is_filterable' => 0,
                    'is_comparable' => 0,
                    'is_visible_on_front' => 0,
                    'used_in_product_listing' => 0,
                    'is_html_allowed_on_front' => 1,
                    'is_unique' => 0,
                    'is_used_for_price_rules' => 0,
                    'is_filterable_in_search' => 0,
                    'used_in_product_listing' => 0,
                    'used_for_sort_by' => 0,
                    'is_visible_in_advanced_search' => 1,
                    'is_wysiwyg_enabled' => 0,
                    'is_used_for_promo_rules' => 0,
                    'is_required_in_admin_store' => 0,
                    'is_used_in_grid' => 0,
                    'is_visible_in_grid' => 0,
                    'is_filterable_in_grid' => 0
                ];

                if (!empty($optionsData) && !empty($optionsData[$rowAttribute])) {
                    $attributes[$rowAttribute]['option']['value'] = $optionsData[$rowAttribute];
                }
            }

            if (\Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE == $behavior) {
                if ($listAttributes) {
                    if ($this->deleteAttributes(array_unique($listAttributes))) {
                        $this->saveAttributes($attributes);
                    }
                }
            } elseif (\Magento\ImportExport\Model\Import::BEHAVIOR_APPEND == $behavior) {
                $this->saveAttributes($attributes);
            }

        }

        return $this;
    }

    /**
     * Save attributes
     *
     * @param array $attributeDatas
     */
    public function saveAttributes(array $attributeDatas)
    {
        if ($attributeDatas) {
            foreach ($attributeDatas as $attributeCode => $attributeData) {
                $attribute = $this->eavAttribute->create();
                $attribute->setData($attributeData);
                try {
                    $this->productAttributeRepository->save($attribute);
                    $this->addAttributeToAllAttributeSet($attributeCode);
                } catch (\Magento\Framework\Exception\AlreadyExistsException $ex) {
                    $this->logger->info($ex->getMessage() . ': ' . $attributeCode);
                    $this->addRowError($ex->getMessage(), $this->_processedRowsCount, null, null, ProcessingError::ERROR_LEVEL_NOT_CRITICAL);
                } catch (\Exception $ex) {
                    $this->logger->info($ex->getMessage() . ': ' . $attributeCode);
                    $this->addRowError($ex->getMessage(), $this->_processedRowsCount);
                }
            }
        }
    }

    public function addAttributeToAllAttributeSet($attributeCode) {
        $attributeSetOptions = $this->optionsFactory->create();
        foreach($attributeSetOptions->toOptionArray() as $attributeSetOptions) {
            $attributeSet = $this->attributeSet->get($attributeSetOptions['value']);
            $this->attributeManagement->assign(
                'catalog_product',
                $attributeSetOptions['value'],
                $attributeSet->getDefaultGroupId(),
                $attributeCode,
                $attributeSet->getCollection()->count() * 10
            );
        }
    }

    /**
     * Deletes attributes.
     *
     * @param array $attributeCodes
     */
    public function deleteAttributes(array $attributeCodes)
    {
        foreach ($attributeCodes as $attributeCode) {
            try {
                $this->productAttributeRepository->deleteById($attributeCode);
            } catch (\Exception $ex) {
                $this->logger->info($ex->getMessage());
                $this->addRowError($ex->getMessage(), $this->_processedRowsCount);
            }
        }
    }

    /**
     * Get Store Id By Code
     * @param type $storeCode
     * @return int
     */
    public function getStoreIdByCode($storeCode)
    {
        $stores = $this->storeManager->getStores(true, false);
        foreach ($stores as $store) {
            if ($store->getCode() === $storeCode) {
                return $store->getId();
            }
        }
        return 0;
    }

}
