<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Formatter;

use \Magento\Catalog\Ui\DataProvider\Product\Form\Modifier\Images;
use Magento\CatalogImportExport\Model\Import\Product\Validator\Media;

class Product extends AbstractFormatter {

    const FIELD_PRODUCT_SKU = 'sku';
    const FIELD_PRODUCT_NAME = 'name';
    const FIELD_PRODUCT_DESCRIPTION = 'description';
    const FIELD_PRODUCT_ATTRIBUTE_SET = 'attribute_set_code';
    const FIELD_PRODUCT_PRICE = 'prezzo';

    const DEFAULT_PRODUCT_PRICE = 0;
    const DEFAULT_PRODUCT_STATUS = 'Disabled';
    const DEFAULT_PRODUCT_VISIBILITY = 'Not Visible Individually';
    const DEFAULT_PRODUCT_MANAGE_STOCK = 'No';

    const FIELD_PRODUCT_RELATED = 'related_skus';
    const FIELD_PRODUCT_CROSSSELL = 'crosssell_skus';
    const FIELD_PRODUCT_UPSELL = 'upsell_skus';
    const DEFAULT_RELATED_DIVIDER = ',';

    /**
     * @var \Magento\Catalog\Model\Product
     */
    private $product;
    /**
     * @var \Drop\Import\Helper\Product
     */
    private $helper;

    /**
     * Product constructor.
     * @param \Magento\Catalog\Model\Product $product
     * @param \Drop\Import\Helper\Product $helper
     */
    public function __construct(
        \Magento\Catalog\Model\Product $product,
        \Drop\Import\Helper\Product $helper
    )
    {
        $this->product = $product;
        $this->helper = $helper;
    }

    /**
     * Rewrite this to change product sku
     * @return string
     */
    public function skuFormatter() {
        return strtoupper($this->getStringUtf8($this->getRow()[self::FIELD_PRODUCT_SKU]));
    }

    /**
     * Rewrite this to change product simple sku
     * @param string $addString
     * @return string
     */
    public function simpleAssociatedSkuFormatter(string $addString = null) {
        return strtoupper($this->skuFormatter() . '-' . $this->getStringUtf8($addString));
    }

    /**
     * Rewrite this to change product name
     * @return string
     */
    public function nameFormatter() {
        return $this->getStringUtf8($this->getRow()[$this->getFieldTranslation(self::FIELD_PRODUCT_NAME)]);
    }

    /**
     * Rewrite this to change product simple name
     * @param string $addString
     * @return string
     */
    public function simpleAssociatedNameFormatter(string $addString = null) {
        return $this->nameFormatter() . ' - ' . $this->getStringUtf8($this->getRow()[self::FIELD_PRODUCT_ATTRIBUTE_SET] . $addString);
    }

    /**
     * Rewrite this to change product description
     * @return string
     */
    public function descriptionFormatter() {
        return $this->getStringUtf8($this->getRow()[$this->getFieldTranslation(self::FIELD_PRODUCT_DESCRIPTION)]);
    }

    /**
     * Rewrite this to change product url
     * MUST BE UNIQUE - PLEASE NEVER REMOVE SKU FROM URL
     * @return string
     */
    public function urlFormatter() {
        return $this->product->formatUrlKey($this->nameFormatter() . '-' . $this->skuFormatter());
    }

    /**
     * Rewrite this to change product simple url
     * MUST BE UNIQUE - PLEASE NEVER REMOVE SKU FROM URL
     * @param string $addNameString
     * @param string $addSkuString
     * @return string
     */
    public function simpleAssociatedUrlFormatter(string $addNameString = null, string $addSkuString = null) {
        return $this->product->formatUrlKey($this->simpleAssociatedNameFormatter($addNameString) . '-' . $this->simpleAssociatedSkuFormatter($addSkuString));
    }

    /**
     * Rewrite this to change product meta title
     * @return string
     */
    public function metatitleFormatter() {
        return $this->getStringUtf8($this->getRow()[$this->getFieldTranslation(self::FIELD_PRODUCT_NAME)]);
    }

    /**
     * Rewrite this to change product meta description
     * @return string
     */
    public function metadescriptionFormatter() {
        return $this->getStringUtf8($this->getRow()[$this->getFieldTranslation(self::FIELD_PRODUCT_DESCRIPTION)]);
    }

    /**
     * Rewrite this to change product image name
     *
     * @param int $counter
     * @return string
     */
    public function imageNameFormatter(int $counter) {
        return $this->skuFormatter() . $this->helper->getImageCounterSeparator() . $counter;
    }

    /**
     * Rewrite this to change product image name
     *
     * @param string $addString
     * @param int $counter
     * @return string
     */
    public function swatchImageNameFormatter(string $addString, int $counter) {
        return $this->simpleAssociatedSkuFormatter($addString) . $this->helper->getImageCounterSeparator() . $counter;
    }

    /**
     * Rewrite this to change product image role
     *
     * @param int $i
     * @return array
     */
    public function imageRoleFormatter(int $i)
    {
        switch ($i) {
            case 0:
                return [Images::CODE_IMAGE, Images::CODE_SMALL_IMAGE];
            case 1:
                return [Images::CODE_THUMBNAIL];
        }
        return [Media::ADDITIONAL_IMAGES];
    }

    /**
     * @return string
     */
    public function getProductPrice() {
        return self::DEFAULT_PRODUCT_PRICE;
    }

    /**
     * @return string
     */
    public function getProductStatus() {
        return self::DEFAULT_PRODUCT_STATUS;
    }

    /**
     * @return string
     */
    public function getProductVisibility() {
        return self::DEFAULT_PRODUCT_VISIBILITY;
    }

    /**
     * @return string
     */
    public function getProductManageStock() {
        return self::DEFAULT_PRODUCT_MANAGE_STOCK;
    }

    /**
     * Rewrite this to change related product sku
     * @return string
     */
    public function relatedSkuFormatter() {
        return strtoupper($this->getStringUtf8($this->getRow()[self::FIELD_PRODUCT_RELATED]));
    }

    /**
     * Rewrite this to change crosssell product sku
     * @return string
     */
    public function crosssellSkuFormatter() {
        return strtoupper($this->getStringUtf8($this->getRow()[self::FIELD_PRODUCT_CROSSSELL]));
    }

    /**
     * Rewrite this to change crosssell product sku
     * @return string
     */
    public function upsellSkuFormatter() {
        return strtoupper($this->getStringUtf8($this->getRow()[self::FIELD_PRODUCT_UPSELL]));
    }

    /**
     * @param $relatedTypology
     * @return string
     */
    public function relatedPositionFormatter($relatedTypology) {
        $position = '';
        $positionLimit = substr_count($this->getRow()[$relatedTypology], self::DEFAULT_RELATED_DIVIDER) + 1;
        for($positionCounter = 1; $positionCounter<=$positionLimit; $positionCounter++) {
            $position .= $positionCounter . ',';
        }
        return rtrim($position, ',');
    }

    public function priceFormatter() {
        $price = str_replace(',', '.', $this->getRow()[self::FIELD_PRODUCT_PRICE]);
        $price = preg_replace("/[^0-9\.]/", "", $price);    // remove everything except numbers and dot "."
        return number_format((float) $price, 4); //Convert to 4 digits
    }
}
