<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Checkers\Entities\Product\Update;

use Drop\Import\Model\Checkers\CheckerInterface;
use \Drop\Import\Model\Checkers\Entities\Product;

class Image extends Product implements CheckerInterface {


    /**
     * @var \Drop\Import\Helper\Product
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Formatter\Product
     */
    private $formatter;
    /**
     * @var \Drop\Import\Model\AttributeFactory
     */
    private $attributeFactory;

    /**
     * Product constructor.
     * @param \Drop\Import\Helper\Product $helper
     * @param \Drop\Import\Model\Formatter\Product\Swatch $formatter
     * @param \Drop\Import\Model\AttributeFactory $attributeFactory
     */
    public function __construct(
        \Drop\Import\Helper\Product $helper,
        \Drop\Import\Model\Formatter\Product\Swatch $formatter,
        \Drop\Import\Model\AttributeFactory $attributeFactory
    )
    {
        $this->helper = $helper;
        $this->formatter = $formatter;
        $this->attributeFactory = $attributeFactory;

        parent::__construct($helper, $formatter, $attributeFactory);
    }

    public function validateData(array $inputData)
    {
        $validImages = [];
        foreach($inputData as $key => $row) {

            $this->formatter->setRow($row);

            if(empty($this->formatter->skuFormatter())) {
                $this->addErrorMessage('Empty sku');
            } else {
                $this->getBehaviorExistanceErrors();
            }

            // Product images
            $validImagesConsecutive = [];
            for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                foreach($this->helper->getImageSupportedFormats() as $format) {
                    $imageName = $this->formatter->imageNameFormatter($imagesCounter);
                    if ($this->helper->getImageExistBySkuAndCounter($imageName . $format)) {
                        $validImages[$this->formatter->skuFormatter()][] = $this->helper->getRelativeImportImagesFileDir() . $imageName . $format;
                        $validImagesConsecutive[] = $imagesCounter;
                    }
                }
            }

            if(!isset($validImages[$this->formatter->skuFormatter()]) || (!count($validImages[$this->formatter->skuFormatter()]))) {
                $this->addErrorMessage("At least 1 image is required");
            } elseif (!$this->helper->getArrayIsConsecutive($validImagesConsecutive)) {
                $this->addErrorMessage("Images counter are not consecutive " . implode(',' , $validImagesConsecutive));
            }

            $swatchVariation = $this->formatter->swatchVariationFormatter();
            if(empty($swatchVariation)) {
                $this->addWarningMessage('Empty swatch_variation. Images for swatches will not be imported.');
                continue;
            }
            $swatchesOptions = $this->attributeFactory->create()->getSwatchAttributeOptions($row['attribute_set_code']);
            if(empty($swatchesOptions)) {
                $this->addErrorMessage('Empty swatch_options for attribute_set_code');
            }

            $validImagesConsecutive = [];
            foreach($swatchesOptions as $swatchAttributeCode => $swatchOptions) {
                foreach($swatchOptions as $swatchOption) {
                    if (empty($swatchOption['value']) || empty($swatchOption['label'])) {
                        continue;
                    }
                    if(!in_array($swatchOption['label'], $swatchVariation)) {
                        continue;
                    }

                    for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                        foreach($this->helper->getImageSupportedFormats() as $format) {
                            $imageName = $this->formatter->swatchImageNameFormatter($swatchOption['label'], $imagesCounter);
                            if ($this->helper->getImageExistBySkuAndCounter($imageName . $format)) {
                                $validImages[$this->formatter->simpleAssociatedSkuFormatter($swatchOption['label'])][] = $this->helper->getRelativeImportImagesFileDir() . $imageName . $format;
                                $validImagesConsecutive[$swatchOption['label']][] = $imagesCounter;
                            }
                        }
                    }
                }
            }

            foreach($validImagesConsecutive as $sku => $validImageConsecutive) {
                if(!isset($validImageConsecutive) || (!count($validImageConsecutive))) {
                    $this->addErrorMessage("At least 1 image for swatches variation is required");
                } elseif (!$this->helper->getArrayIsConsecutive($validImageConsecutive)) {
                    $this->addErrorMessage("Images counter for swatches variation are not consecutive " . implode(',' , $validImageConsecutive));
                }
            }

            $this->setRowIncrement();
        }

        $this->helper->setProductImagesInformation($validImages);

        return $this;
    }

}
