<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Checkers\Entities\Product;

use Drop\Import\Model\Checkers\CheckerInterface;
use \Drop\Import\Model\Checkers\Entities\Product;

class Swatch extends Product implements CheckerInterface {


    /**
     * @var \Drop\Import\Helper\Product
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Formatter\Product
     */
    private $formatter;
    /**
     * @var \Drop\Import\Model\AttributeFactory
     */
    private $attributeFactory;

    /**
     * Product constructor.
     * @param \Drop\Import\Helper\Product $helper
     * @param \Drop\Import\Model\Formatter\Product\Swatch $formatter
     * @param \Drop\Import\Model\AttributeFactory $attributeFactory
     */
    public function __construct(
        \Drop\Import\Helper\Product $helper,
        \Drop\Import\Model\Formatter\Product\Swatch $formatter,
        \Drop\Import\Model\AttributeFactory $attributeFactory
    )
    {
        $this->helper = $helper;
        $this->formatter = $formatter;
        $this->attributeFactory = $attributeFactory;

        parent::__construct($helper, $formatter, $attributeFactory);
    }

    public function validateData(array $inputData)
    {
        $translationLanguages = $this->getTranslationLanguage();
        $validImages = [];
        foreach($inputData as $key => $row) {

            $this->formatter->setRow($row);

            if(empty($this->formatter->skuFormatter())) {
                $this->addErrorMessage('Empty sku');
            } else {
                $this->getBehaviorExistanceErrors();
            }

            if(empty($this->formatter->nameFormatter())) {
                $this->addErrorMessage('Empty name');
            }
            if(empty($this->formatter->descriptionFormatter())) {
                $this->addErrorMessage('Empty description');
            }
            if(empty($row['weight'])) {
                $this->addErrorMessage('Empty weight');
            }
            if(empty($this->formatter->metatitleFormatter())) {
                $this->addErrorMessage('Empty meta_title');
            } elseif(strlen($this->formatter->metatitleFormatter()) > self::DEFAULT_META_TITLE_LENGTH) {
                $this->addErrorMessage('Max meta_title length is ' . self::DEFAULT_META_TITLE_LENGTH . ' character');
            }
            if(empty($this->formatter->metadescriptionFormatter())) {
                $this->addErrorMessage('Empty meta_description');
            } elseif(strlen($this->formatter->metadescriptionFormatter()) > self::DEFAULT_META_DESCRIPTION_LENGTH) {
                $this->addErrorMessage('Max meta_description length is ' . self::DEFAULT_META_DESCRIPTION_LENGTH . ' character');
            }

            foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code']) as $userDefinedAttribute) {
                if(empty($row[$userDefinedAttribute])) {
                    $this->addErrorMessage("Empty $userDefinedAttribute");
                }
            }

            if(empty($row['websites'])) {
                $this->addErrorMessage('Empty websites');
            } else {
                $websiteCodes = explode(',', $row['websites']);
                foreach($websiteCodes as $websiteCode) {
                    if(!$this->helper->getWebsiteExistByCode($websiteCode)) {
                        $this->addErrorMessage('websites invalid');
                    }
                }
            }

            if(empty($row['categories'])) {
                $this->addErrorMessage('Empty categories');
            } else {
                $categories = explode($this->helper->getCategoryHelper()->getCategoryPathSeparator(), $row['categories']);
                if(!$this->helper->getCategoryHelper()->getCategoryTreeExist($categories)) {
                    $this->addErrorMessage('Category Tree invalid');
                }
            }

            if(empty($row['attribute_set_code'])) {
                $this->addErrorMessage('Empty attribute_set_code');
            } else {
                if(!$this->helper->getAttributeSetHelper()->getAttributeSetExistByName($row['attribute_set_code'])) {
                    $this->addErrorMessage('Invalid attribute_set_code');
                }
            }

            $superAttributeOptions = $this->attributeFactory->create()->getSuperAttributeOptions($row['attribute_set_code']);
            if(empty($superAttributeOptions)) {
                $this->addErrorMessage('Empty super_attribute for attribute_set_code');
            }

            $swatchesOptions = $this->attributeFactory->create()->getSwatchAttributeOptions($row['attribute_set_code']);
            if(empty($swatchesOptions)) {
                $this->addErrorMessage('Empty swatch_options for attribute_set_code');
            }
            $swatchVariation = $this->formatter->swatchVariationFormatter();
            if(empty($swatchVariation)) {
                $this->addErrorMessage('Empty swatch_variation');
            }

            // Product images
            $validImagesConsecutive = [];
            for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                foreach($this->helper->getImageSupportedFormats() as $format) {
                    $imageName = $this->formatter->imageNameFormatter($imagesCounter);
                    if ($this->helper->getImageExistBySkuAndCounter($imageName . $format)) {
                        $validImages[$this->formatter->skuFormatter()][] = $this->helper->getRelativeImportImagesFileDir() . $imageName . $format;
                        $validImagesConsecutive[] = $imagesCounter;
                    }
                }
            }

            if(!isset($validImages[$this->formatter->skuFormatter()]) || (!count($validImages[$this->formatter->skuFormatter()]))) {
                $this->addErrorMessage("At least 1 image is required");
            } elseif (!$this->helper->getArrayIsConsecutive($validImagesConsecutive)) {
                $this->addErrorMessage("Images counter are not consecutive " . implode(',' , $validImagesConsecutive));
            }

            $validImagesConsecutive = [];
            foreach($swatchesOptions as $swatchAttributeCode => $swatchOptions) {
                foreach($swatchOptions as $swatchOption) {
                    if (empty($swatchOption['value']) || empty($swatchOption['label'])) {
                        continue;
                    }
                    if(!in_array($swatchOption['label'], $swatchVariation)) {
                        continue;
                    }

                    for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                        foreach($this->helper->getImageSupportedFormats() as $format) {
                            $imageName = $this->formatter->swatchImageNameFormatter($swatchOption['label'], $imagesCounter);
                            if ($this->helper->getImageExistBySkuAndCounter($imageName . $format)) {
                                $validImages[$this->formatter->simpleAssociatedSkuFormatter($swatchOption['label'])][] = $this->helper->getRelativeImportImagesFileDir() . $imageName . $format;
                                $validImagesConsecutive[$swatchOption['label']][] = $imagesCounter;
                            }
                        }
                    }
                }
            }

            foreach($validImagesConsecutive as $sku => $validImageConsecutive) {
                if(!isset($validImageConsecutive) || (!count($validImageConsecutive))) {
                    $this->addErrorMessage("At least 1 image for swatches variation is required");
                } elseif (!$this->helper->getArrayIsConsecutive($validImageConsecutive)) {
                    $this->addErrorMessage("Images counter for swatches variation are not consecutive " . implode(',' , $validImageConsecutive));
                }
            }

            foreach($translationLanguages as $language) {
                if($language == $this->helper->getReferenceStoreCode()) {
                    continue;
                }

                if(empty($this->formatter->nameFormatter())) {
                    $this->addErrorMessage('Empty name_' . $language);
                }
                if(empty($this->formatter->descriptionFormatter())) {
                    $this->addErrorMessage('Empty description_' . $language);
                }

                foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code'], $language) as $userDefinedAttribute) {
                    if(empty($row[$userDefinedAttribute])) {
                        $this->addErrorMessage("Empty $userDefinedAttribute");
                    }
                }

                $this->formatter->setLanguage($language);
            }

            $this->setRowIncrement();
            $this->formatter->resetLanguage();
        }

        $this->helper->setProductImagesInformation($validImages);

        return $this;
    }

}
