<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Checkers\Entities;

use \Drop\Import\Model\Checkers\AbstractChecker;
use Drop\Import\Model\Checkers\CheckerInterface;

class Product extends AbstractChecker implements CheckerInterface {
    
    /**
     * @var \Drop\Import\Helper\Product
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Formatter\Product
     */
    private $formatter;
    /**
     * @var \Drop\Import\Model\AttributeFactory
     */
    private $attributeFactory;

    /**
     * Product constructor.
     * @param \Drop\Import\Helper\Product $helper
     * @param \Drop\Import\Model\Formatter\Product $formatter
     * @param \Drop\Import\Model\AttributeFactory $attributeFactory
     */
    public function __construct(
        \Drop\Import\Helper\Product $helper,
        \Drop\Import\Model\Formatter\Product $formatter,
        \Drop\Import\Model\AttributeFactory $attributeFactory
    )
    {
        $this->helper = $helper;
        $this->formatter = $formatter;
        $this->attributeFactory = $attributeFactory;
    }

    public function validateData(array $inputData)
    {
        $translationLanguages = $this->getTranslationLanguage();
        $validImages = [];
        foreach($inputData as $key => $row) {

            $this->formatter->setRow($row);

            if(empty($this->formatter->skuFormatter())) {
                $this->addErrorMessage('Empty sku');
            } else {
                $this->getBehaviorExistanceErrors();
            }

            if(empty($this->formatter->nameFormatter())) {
                $this->addErrorMessage('Empty name');
            }
            if(empty($this->formatter->descriptionFormatter())) {
                $this->addErrorMessage('Empty description');
            }
            if(empty($row['weight'])) {
                $this->addErrorMessage('Empty weight');
            }
            if(empty($this->formatter->metatitleFormatter())) {
                $this->addErrorMessage('Empty meta_title');
            } elseif(strlen($this->formatter->metatitleFormatter()) > self::DEFAULT_META_TITLE_LENGTH) {
                $this->addErrorMessage('Max meta_title length is ' . self::DEFAULT_META_TITLE_LENGTH . ' character');
            }
            if(empty($this->formatter->metadescriptionFormatter())) {
                $this->addErrorMessage('Empty meta_description');
            } elseif(strlen($this->formatter->metadescriptionFormatter()) > self::DEFAULT_META_DESCRIPTION_LENGTH) {
                $this->addErrorMessage('Max meta_description length is ' . self::DEFAULT_META_DESCRIPTION_LENGTH . ' character');
            }

            if(empty($row['websites'])) {
                $this->addErrorMessage('Empty websites');
            } else {
                $websiteCodes = explode(',', $row['websites']);
                foreach($websiteCodes as $websiteCode) {
                    if(!$this->helper->getWebsiteExistByCode($websiteCode)) {
                        $this->addErrorMessage('websites invalid');
                    }
                }
            }

            if(empty($row['categories'])) {
                $this->addErrorMessage('Empty categories');
            } else {
                $categories = explode($this->helper->getCategoryHelper()->getCategoryPathSeparator(), $row['categories']);
                if(!$this->helper->getCategoryHelper()->getCategoryTreeExist($categories)) {
                    $this->addErrorMessage('Category Tree invalid');
                }
            }
            
            if(empty($row['attribute_set_code'])) {
                $this->addErrorMessage('Empty attribute_set_code');
            } else {
                if(!$this->helper->getAttributeSetHelper()->getAttributeSetExistByName($row['attribute_set_code'])) {
                    $this->addErrorMessage('Invalid attribute_set_code');
                }
            }

            foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code']) as $userDefinedAttribute) {
                if(empty($row[$userDefinedAttribute])) {
                    $this->addErrorMessage("Empty $userDefinedAttribute");
                }
            }

            $superAttributeOptions = $this->attributeFactory->create()->getSuperAttributeOptions($row['attribute_set_code']);
            if(empty($superAttributeOptions)) {
                $this->addErrorMessage('Empty super_attribute for attribute_set_code');
            }

            // Product images
            $validImagesConsecutive = [];
            for ($imagesCounter = 1; $imagesCounter < 10; $imagesCounter++) {
                foreach($this->helper->getImageSupportedFormats() as $format) {
                    $imageName = $this->formatter->imageNameFormatter($imagesCounter);
                    if ($this->helper->getImageExistBySkuAndCounter($imageName . $format)) {
                        $validImages[$this->formatter->skuFormatter()][] = $this->helper->getRelativeImportImagesFileDir() . $imageName . $format;
                        $validImagesConsecutive[] = $imagesCounter;
                    }
                }
            }

            if(!isset($validImages[$this->formatter->skuFormatter()]) || (!count($validImages[$this->formatter->skuFormatter()]))) {
                $this->addErrorMessage("At least 1 image is required");
            } elseif (!$this->helper->getArrayIsConsecutive($validImagesConsecutive)) {
                $this->addErrorMessage("Images counter are not consecutive " . implode(',' , $validImagesConsecutive));
            }

            foreach($translationLanguages as $language) {
                if($language == $this->helper->getReferenceStoreCode()) {
                    continue;
                }

                if(empty($this->formatter->nameFormatter())) {
                    $this->addErrorMessage('Empty name_' . $language);
                }
                if(empty($this->formatter->descriptionFormatter())) {
                    $this->addErrorMessage('Empty description_' . $language);
                }

                foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code'], $language) as $userDefinedAttribute) {
                    if(empty($row[$userDefinedAttribute])) {
                        $this->addErrorMessage("Empty $userDefinedAttribute");
                    }
                }

                $this->formatter->setLanguage($language);
            }

            $this->setRowIncrement();
            $this->formatter->resetLanguage();
        }

        $this->helper->setProductImagesInformation($validImages);

       return $this;
    }

    /**
     * Check product existance based on behavior admin configuration
     */
    protected function getBehaviorExistanceErrors() {
        $behavior = $this->helper->getBehavior();

        switch ($behavior) {
            case \Magento\ImportExport\Model\Import::BEHAVIOR_APPEND:
                if($this->helper->getProductExistBySku($this->formatter->skuFormatter())) {
                    $this->addErrorMessage("Product exist for Behavior $behavior");
                }
                break;
            case \Drop\Import\Model\Config\Source\Behavior::BEHAVIOR_UPDATE:
            case \Magento\ImportExport\Model\Import::BEHAVIOR_REPLACE:
            case \Magento\ImportExport\Model\Import::BEHAVIOR_DELETE:
                if(!$this->helper->getProductExistBySku($this->formatter->skuFormatter())) {
                    $this->addErrorMessage("Product not exist for Behavior $behavior");
                }
                break;
        }
    }

}
