<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Model\Checkers\Entities;

use \Drop\Import\Model\Checkers\AbstractChecker;
use Drop\Import\Model\Checkers\CheckerInterface;

class CustomerComposite extends AbstractChecker implements CheckerInterface {
    
    /**
     * @var \Drop\Import\Helper\Customer
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Formatter\Customer
     */
    private $formatter;

    /**
     * CustomerComposite constructor.
     * @param \Drop\Import\Helper\Customer $helper
     * @param \Drop\Import\Model\Formatter\Customer $formatter
     */
    public function __construct(
        \Drop\Import\Helper\Customer $helper,
        \Drop\Import\Model\Formatter\Customer $formatter
    )
    {
        $this->helper = $helper;
        $this->formatter = $formatter;
    }

    public function validateData(array $inputData)
    {
        foreach($inputData as $key => $row) {

            $this->formatter->setRow($row);

            if(empty($row['email'])) {
                $this->addErrorMessage('Email empty');
            }
            if(!$this->helper->getEmailIsValid($row['email'])) {
                $this->addErrorMessage('Email invalid');
            }
            if(empty($row['firstname'])) {
                $this->addErrorMessage('Empty firstname');
            }
            if(empty($row['lastname'])) {
                $this->addErrorMessage('Empty lastname');
            }
            if(empty($this->formatter->ipAddressFormatter())) {
                $this->addErrorMessage('Empty ip_address');
            }
            if(!$this->helper->getIpIsValid($row['ip_address'])) {
                $this->addErrorMessage('Ip Address invalid');
            }
            if(empty($this->formatter->privacyFormatter())) {
                $this->addErrorMessage('Empty privacy_policy');
            }
            if(empty($this->formatter->profilingFormatter())) {
                $this->addErrorMessage('Empty profiling');
            }

            if(empty($row['country']) && empty($row['website_id']) && empty($row['website_code'])) {
                $this->addErrorMessage('Empty one of country/website_id/website_code');
            } else {
                if(isset($row['country'])) {
                    if(!$this->helper->getCountryExistByCode($row['country'])){
                        $this->addErrorMessage('country is not a valid field');
                    } else {
                        $row['website_id'] = $this->helper->getWebsiteByCountry($row['country']);
                    }
                }
                if(isset($row['website_id'])) {
                    if(!$this->helper->getWebsiteExistById($row['website_id'])) {
                        $this->addErrorMessage('website_id is not a valid field');
                    }
                }
                if(isset($row['website_code'])) {
                    if(!$this->helper->getWebsiteExistByCode($row['website_code'])) {
                        $this->addErrorMessage('website_code is not a valid field');
                    } else {
                        $row['website_id'] = $this->helper->getWebsiteIdByCode($row['website_code']);
                    }
                }

                if($this->helper->getCustomerExistByEmail($row['email'], $row['website_id'])) {
                    $this->addErrorMessage('Customer already exist');
                }

            }

            if(empty($this->formatter->genderFormatter())) {
                $this->addErrorMessage('Empty gender');
            }
            if(empty($this->formatter->privacyFormatter())) {
                $this->addErrorMessage('Empty privacy_policy');
            }
            if(empty($this->formatter->profilingFormatter())) {
                $this->addErrorMessage('Empty profiling');
            }

            if(empty($row['street'])) {
                $this->addErrorMessage('Empty street');
            }
            if(empty($row['city'])) {
                $this->addErrorMessage('Empty city');
            }

            if(empty($row['region'])) {
                $this->addErrorMessage('Empty region');
            }
            if(!$this->helper->getRegionExistbyName($row['region'], $row['country'])) {
                $this->addErrorMessage('region invalid');
            }
            if(empty($row['postcode'])) {
                $this->addErrorMessage('Empty postcode');
            }
            if(empty($row['country'])) {
                $this->addErrorMessage('Empty country');
            }
            if(!$this->helper->getCountryExistByCode($row['country'])) {
                $this->addErrorMessage('country invalid');
            }

            if(!$this->helper->getGeolocalizeAddress()) {
                $this->setRowIncrement();
                continue;
            }
            if(!$this->helper->geocodeModuleExist()) {
                $this->addErrorMessage('Geocoding module does not exist. Disable geolocalize setting or install limesharp/stockists');
                $this->setRowIncrement();
                continue;
            }

            $address = $this->formatter->addressFormatter();
            $storeAddress = $this->formatter->storeAddressFormatter($address);
            $geocodedStoreAddress = $this->helper->geocodeAddress($storeAddress);

            //TODO: Not tested from here
//            if(empty($geocodedStoreAddress['administrative_area_level_2']) || !$this->helper->getRegionExistbyName($geocodedStoreAddress['administrative_area_level_2'], $row['country'])) {
//                $this->addErrorMessage('administrative_area_level_2 not empty and not found in magento');
//            } elseif(empty($geocodedStoreAddress['administrative_area_level_1']) || !$this->helper->getRegionExistbyName($geocodedStoreAddress['administrative_area_level_1'], $row['country'])) {
//                $this->addErrorMessage('administrative_area_level_1 not empty and not found in magento');
//            } else {
//                $this->addErrorMessage('administrative_area_level_1 and administrative_area_level_2 are empty');
//            }
//
//            if(empty($geocodedStoreAddress['country']) || empty($geocodedStoreAddress['route']) || empty($geocodedStoreAddress['street_number']) || empty($geocodedStoreAddress['locality']) || empty($geocodedStoreAddress['postal_code'])) {
//                $this->addErrorMessage('empty necessary google field');
//            }

            $this->setRowIncrement();
        }

       return $this;
    }

}
