<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Helper;

use Magento\Framework\Exception\NoSuchEntityException;

class Product extends Data implements EntitiesInterface {

    const ENTITY_CODE = 'catalog_product';

    const XML_PATH_PRODUCT_BEHAVIOR            = 'dropimport/product_configuration/behavior';
    const XML_PATH_ATTRIBUTE_IMPORT_CRON           = 'dropimport/product_configuration/import_cron';

    const XML_PATH_PRODUCT_IMPORT_METHOD = 'dropimport/product_configuration/import_method';
    const XML_PATH_PRODUCT_IMPORT_IMAGES_FILE_DIR = 'dropimport/product_configuration/import_images_file_dir';

    const XML_PATH_PRODUCT_IMPORT_SPREADSHEET_ID = 'dropimport/product_configuration/spreadsheet_id';
    const XML_PATH_PRODUCT_IMPORT_SHEET_TITLE = 'dropimport/product_configuration/import_sheet_title';
    const XML_PATH_PRODUCT_IMPORT_PRICE_SHEET_TITLE = 'dropimport/product_configuration/import_price_sheet_title';
    const XML_PATH_PRODUCT_IMPORT_STOCK_SHEET_TITLE = 'dropimport/product_configuration/import_stock_sheet_title';
    const XML_PATH_PRODUCT_IMPORT_RELATED_SHEET_TITLE = 'dropimport/product_configuration/import_related_sheet_title';

    const XML_PATH_PRODUCT_IMPORT_PATH_URL = 'dropimport/product_configuration/import_path_url';
    const XML_PATH_PRODUCT_IMPORT_PRICE_PATH_URL = 'dropimport/product_configuration/import_price_path_url';
    const XML_PATH_PRODUCT_IMPORT_STOCK_PATH_URL = 'dropimport/product_configuration/import_stock_path_url';
    const XML_PATH_PRODUCT_IMPORT_RELATED_PATH_URL = 'dropimport/product_configuration/import_related_path_url';

    const SHEET_TITLE_PRICE = 'price';
    const SHEET_TITLE_STOCK = 'stock';
    const SHEET_TITLE_RELATED = 'related';

    const IMAGES_COUNTER_SEPARATOR = '_';
    private $imageFormats = [
        '.jpg',
        '.png'
    ];
    protected $productImagesInformation = [];
    protected $behavior;
    protected $sheetTitle;

    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    private $directoryList;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface\Proxy
     */
    private $productRepository;

    /**
     * Product constructor.
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\Config\Storage\WriterInterface $configWriter
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory
     * @param \Magento\Directory\Model\CountryFactory $countryFactory
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     * @param \Magento\Catalog\Api\ProductRepositoryInterface\Proxy $productRepository
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\Config\Storage\WriterInterface $configWriter,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        \Magento\Framework\App\Filesystem\DirectoryList $directoryList,
        \Magento\Catalog\Api\ProductRepositoryInterface\Proxy $productRepository
    )
    {
        $this->directoryList = $directoryList;
        $this->productRepository = $productRepository;
        parent::__construct($context, $scopeConfig, $configWriter, $storeManager, $customerGroupFactory, $countryFactory);
    }

    /**
     * Get Behavior
     *
     * @return string
     */
    public function getBehavior()
    {
        if(!empty($this->behavior)) {
            return $this->behavior;
        }
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_BEHAVIOR);
    }

    /**
     * @param $behavior
     * @return $this
     */
    public function setBehavior($behavior) {
        $this->behavior = $behavior;
        return $this;
    }

    /**
     * Get Import Cron
     *
     * @return int
     */
    public function getImportCron()
    {
        return $this->getScopeConfig(self::XML_PATH_ATTRIBUTE_IMPORT_CRON);
    }

    /**
     * Get Import Method
     *
     * @return int
     */
    public function getImportMethod()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_METHOD);
    }

    /**
     * Get Spreadsheet Id
     *
     * @return int
     */
    public function getSpreadsheetId()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_SPREADSHEET_ID);
    }

    /**
     * Get Sheet Title
     *
     * @return int
     */
    public function getSheetTitle()
    {
        if(!empty($this->sheetTitle)) {
            switch($this->sheetTitle) {
                case self::SHEET_TITLE_PRICE:
                    return $this->getPriceSheetTitle();
                case self::SHEET_TITLE_STOCK:
                    return $this->getStockSheetTitle();
                case self::SHEET_TITLE_RELATED:
                    return $this->getRelatedSheetTitle();
            }
        }
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_SHEET_TITLE);
    }

    /**
     * Set Sheet Title
     *
     * @param $sheetTitle
     * @return \Drop\Import\Helper\Product
     */
    public function setSheetTitle($sheetTitle)
    {
        $this->sheetTitle = $sheetTitle;
        return $this;
    }

    /**
     * Get Price Sheet Title
     *
     * @return int
     */
    public function getPriceSheetTitle()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_PRICE_SHEET_TITLE);
    }

    /**
     * Get Stock Sheet Title
     *
     * @return int
     */
    public function getStockSheetTitle()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_STOCK_SHEET_TITLE);
    }

    /**
     * Get Related Sheet Title
     *
     * @return int
     */
    public function getRelatedSheetTitle()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_RELATED_SHEET_TITLE);
    }

    /**
     * Get Import Path or Url
     *
     * @return int
     */
    public function getImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_PATH_URL);
    }

    /**
     * Get Price CSV File path
     *
     * @return int
     */
    public function getPriceImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_PRICE_PATH_URL);
    }

    /**
     * Get Stock CSV File path
     *
     * @return int
     */
    public function getStockImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_STOCK_PATH_URL);
    }

    /**
     * Get Related CSV File path
     *
     * @return int
     */
    public function getRelatedImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_RELATED_PATH_URL);
    }

    /**
     * Get Import Images File Dir
     *
     * @return string
     */
    public function getImportImagesFileDir()
    {
        return $this->getScopeConfig(self::XML_PATH_PRODUCT_IMPORT_IMAGES_FILE_DIR);
    }

    /**
     * Get Absolute Import Images File Dir
     *
     * @return string
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    public function getAbsoluteImportImagesFileDir()
    {
        return $this->directoryList->getPath('media') . '/catalog/product/' . $this->getImportImagesFileDir();
    }

    /**
     * Get Relative Import Images File Dir
     *
     * @return string
     */
    public function getRelativeImportImagesFileDir()
    {
        return $this->getImportImagesFileDir();
    }

    /**
     * @return string
     */
    public function getImageCounterSeparator()
    {
        return self::IMAGES_COUNTER_SEPARATOR;
    }

    /**
     * @return array
     */
    public function getImageSupportedFormats()
    {
        return $this->imageFormats;
    }

    /**
     * @param string $imageName
     * @return bool
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    public function getImageExistBySkuAndCounter(string $imageName) {
        if (file_exists($this->getAbsoluteImportImagesFileDir() . $imageName)) {
            return true;
        }
        return false;
    }

    /**
     * Get Entity Code
     * 
     * @return string
     */
    public function getEntityCode()
    {
        return self::ENTITY_CODE;
    }

    /**
     * From product check will be generated an array that will be used to semplify images's import
     * @param array $productImagesInformation
     * @return $this
     */
    public function setProductImagesInformation(array $productImagesInformation) {
        $this->productImagesInformation = $productImagesInformation;
        return $this;
    }

    /**
     * @return array
     */
    public function getProductImagesInformation() {
        return $this->productImagesInformation;
    }

    /**
     * @param $header
     * @param $attributeSet
     * @param string|null $storeView
     * @return array
     */
    public function getUserDefinedAttributes($header, $attributeSet, $storeView = null)
    {
        if(!empty($storeView)) {
            $storeView = '_' . $storeView;
        }

        /** @var \Drop\Import\Model\Attribute $attributesModel */
        $attributesModel = $this->getObjectManager()->get('\Drop\Import\Model\Attribute');
        $configurableAttributes = $attributesModel->getUserDefinedAttributeByAttributeSet($attributeSet);
        $userDefinedAttributes = [];
        foreach ($header as $field) {
            foreach($configurableAttributes as $attribute) {
                if($attribute . $storeView == $field) {
                    $userDefinedAttributes[] = $field;
                }
            }
        }
        return $userDefinedAttributes;
    }

    public function getProductExistBySku(string $sku) {
        try {
            $this->productRepository->get($sku);
            return true;
        } catch (NoSuchEntityException $e) {} catch (\Exception $e) {}
        return false;
    }

}
