<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Helper;

class Data extends \Magento\Framework\App\Helper\AbstractHelper {
    
    /**
     * \Magento\Framework\App\Config\ScopeConfigInterface
     * 
     * @var type 
     */
    protected $_config;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var \Magento\Customer\Model\ResourceModel\Group\CollectionFactory
     */
    private $customerGroupFactory;

    const XML_PATH_CONFIGURATION_TEST_MODE = 'dropimport/configuration/test_mode';
    const XML_PATH_CONFIGURATION_DRY_RUN_MODE = 'dropimport/configuration/dry_run_mode';
    const XML_PATH_CONFIGURATION_REPORT_EMAIL = 'dropimport/configuration/report_email';
    const XML_PATH_CONFIGURATION_SEND_REPORT_EMAIL = 'dropimport/configuration/send_email_report';
    const XML_PATH_CONFIGURATION_REFERENCE_STORE_ID = 'dropimport/configuration/reference_store_id';

    const XML_PATH_GENERAL_EMAIL = 'trans_email/ident_general';
    const XML_PATH_GENERAL_STORE_INFORMATION_NAME = 'general/store_information/name';
    const XML_PATH_COUNTRY = 'general/country/allow';

    /**
     * Caching system
     */
    private $_allStores;
    private $_allWebsites;
    private $_allCustomerGroups;
    /**
     * @var \Magento\Directory\Model\CountryFactory
     */
    private $countryFactory;
    /**
     * @var \Magento\Framework\App\Config\Storage\WriterInterface
     */
    private $configWriter;

    /**
     * Data constructor.
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\Config\Storage\WriterInterface $configWriter
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory
     * @param \Magento\Directory\Model\CountryFactory $countryFactory
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\Config\Storage\WriterInterface $configWriter,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory,
        \Magento\Directory\Model\CountryFactory $countryFactory
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->configWriter = $configWriter;
        $this->customerGroupFactory = $customerGroupFactory;
        $this->countryFactory = $countryFactory;
    }
    
    /**
     * Get object manager
     *
     * @return \Magento\Framework\App\ObjectManager
     */
    public function getObjectManager()
    {
        return \Magento\Framework\App\ObjectManager::getInstance();
    }    
    
    /**
     * Get system product helper
     *
     * @return \Drop\Import\Helper\Product
     */    
    public function getProductHelper() {
        return $this->getObjectManager()->get(\Drop\Import\Helper\Product::class);
    }
    
    /**
     * Get system category helper
     *
     * @return \Drop\Import\Helper\Category
     */    
    public function getCategoryHelper() {
        return $this->getObjectManager()->get(\Drop\Import\Helper\Category::class);
    }

    /**
     * Get system attribute helper
     *
     * @return \Drop\Import\Helper\System\}
     */
    public function getAttributeHelper() {
        return $this->getObjectManager()->get(\Drop\Import\Helper\Attribute::class);
    }

    /**
     * Get system attribute set helper
     *
     * @return \Drop\Import\Helper\AttributeSet
     */
    public function getAttributeSetHelper() {
        return $this->getObjectManager()->get(\Drop\Import\Helper\AttributeSet::class);
    }

    /**
     * Get system customer helper
     *
     * @return \Drop\Import\Helper\Customer
     */
    public function getCustomerHelper()
    {
        return $this->getObjectManager()->get(\Drop\Import\Helper\Customer::class);
    }

    /**
     * Retrieve scope config data
     *
     * @param $path
     * @param string $scopeType
     * @param null $scopeCode
     * @return string
     */
    public function getScopeConfig(
        $path,
        $scopeType = \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
        $scopeCode = null
    ) {
        return $this->scopeConfig->getValue($path, $scopeType, $scopeCode);
    }

    public function setScopeConfig(
        $path,
        $value,
        $scopeType = \Magento\Framework\App\Config\ScopeConfigInterface::SCOPE_TYPE_DEFAULT,
        $scopeId = null
    ) {
        $this->configWriter->save($path, $value, $scopeType, $scopeId = 0);
    }

    /**
     * Get Test Mode is Enabled
     *
     * @return int
     */
    public function getTestModeIsEnabled()
    {
        return $this->getScopeConfig(self::XML_PATH_CONFIGURATION_TEST_MODE);
    }

    /**
     * Get Test Mode is Enabled
     *
     * @return int
     */
    public function getDryRunIsEnabled()
    {
        return $this->getScopeConfig(self::XML_PATH_CONFIGURATION_DRY_RUN_MODE);
    }

    /**
     * Get Test Mode is Enabled
     *
     * @return int
     */
    public function getReferenceStoreId()
    {
        return $this->getScopeConfig(self::XML_PATH_CONFIGURATION_REFERENCE_STORE_ID);
    }

    /**
     * Get Test Mode is Enabled
     *
     * @return int
     */
    public function getReferenceStoreCode()
    {
        return $this->getStoreCodeById($this->getReferenceStoreId());
    }

    /**
     * Send Report Email
     *
     * @return int
     */
    public function getSendEmailReport()
    {
        return $this->getScopeConfig(self::XML_PATH_CONFIGURATION_SEND_REPORT_EMAIL);
    }

    /**
     * Get Report Email
     *
     * @return int
     */
    public function getEmailReport()
    {
        $to = $this->getScopeConfig(self::XML_PATH_CONFIGURATION_REPORT_EMAIL);
        if(strpos($to, ';') !== false) {
            $to = explode(';', $to);
        }
        return $to;
    }

    /**
     * Get Email Sender Data
     *
     * @return int
     */
    public function getFromSender()
    {
        return $this->getScopeConfig(self::XML_PATH_GENERAL_EMAIL);
    }

    /**
     * Get Store Name
     *
     * @return int
     */
    public function getStoreName()
    {
        return $this->getScopeConfig(self::XML_PATH_GENERAL_STORE_INFORMATION_NAME);
    }

    /**
     * Retrieve file extension
     *
     * @param string $fileName
     * @return string
     */
    public function getFileExtension(string $fileName) {
        return substr(strrchr($fileName, '.'), 1);
    }

    /**
     * Get store code of field name
     * Example: categoria_it return it
     * @param string $fieldName
     * @return string|bool
     */
    public function getStoreCodeFromFieldName(string $fieldName) {
        if(strrpos($fieldName, '_') !== false) {
            $codeString = substr($fieldName, strrpos($fieldName, '_') + 1);
            return ($this->getStoreExistByCode($codeString)) ? $codeString : false;
        }
    }

    /**
     * Get Store Id By Code
     * @param string $storeCode
     * @return bool
     */
    public function getStoreExistByCode(string $storeCode) {
        foreach($this->getAllStores() as $store) {
            if($store->getCode() === $storeCode) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get Store Exist By Id
     * @param int $storeId
     * @return bool
     */
    public function getStoreExistById(int $storeId) {
        foreach($this->getAllStores() as $store) {
            if((int) $store->getId() === $storeId) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get Website Exist By Code
     * @param string $websiteCode
     * @return bool
     */
    public function getWebsiteExistByCode(string $websiteCode) {
        foreach($this->getAllWebsites() as $website) {
            if($website->getCode() === $websiteCode) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get Website Exist By Id
     * @param int $websiteId
     * @return bool
     */
    public function getWebsiteExistById(int $websiteId) {
        foreach($this->getAllWebsites() as $website) {
            if((int) $website->getId() === $websiteId) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param string $countryCode
     * @return bool
     */
    public function getCountryExistByCode(string $countryCode){
        $country = $this->countryFactory->create()->loadByCode($countryCode);
        if($country->getId()) {
            return true;
        }
        return false;
    }

    /**
     * Get base website id
     * @return mixed
     */
    public function getBaseWebsiteId() {
        $websites = $this->getAllWebsites();
        foreach ($websites as $website) {
            //If base already exist return 'base'
            if($website->getCode() == 'base') {
                return $website->getWebsiteId();
            }
        }
        //If base doesn't exist return the first
        foreach ($websites as $website) {
            return $website->getWebsiteId();
        }
    }

    /**
     * Get base website code
     * @return string
     */
    public function getBaseWebsiteCode() {
        $websites = $this->getAllWebsites();
        foreach ($websites as $website) {
            //If base already exist return 'base'
            if($website->getCode() == 'base') {
                return $website->getCode();
            }
        }
        //If base doesn't exist return the first
        foreach ($websites as $website) {
            return $website->getCode();
        }
    }

    /**
     * Get default store id
     * @return mixed
     */
    public function getDefaultStore() {
        $stores = $this->getAllStores();
        foreach ($stores as $store) {
            //If $store already exist return 'default'
            if($store->getCode() == 'default') {
                return $store;
            }
        }
        //If default doesn't exist return the first
        foreach ($stores as $store) {
            return $store;
        }
    }

    /**
     * Get default store id
     * @return mixed
     */
    public function getDefaultStoreId() {
        $stores = $this->getAllStores();
        foreach ($stores as $store) {
            //If $store already exist return 'default'
            if($store->getCode() == 'default') {
                return $store->getStoreId();
            }
        }
        //If default doesn't exist return the first
        foreach ($stores as $store) {
            return $store->getStoreId();
        }
    }

    /**
     * Get default store code
     * @return mixed
     */
    public function getDefaultStoreCode() {
        $stores = $this->getAllStores();
        foreach ($stores as $store) {
            //If $store already exist return 'default'
            if($store->getCode() == 'default') {
                return $store->getCode();
            }
        }
        //If default doesn't exist return the first
        foreach ($stores as $store) {
            return $store->getCode();
        }
    }

    /**
     * Get default store label
     * @return mixed
     */
    public function getDefaultStoreLabel() {
        $stores = $this->getAllStores();
        foreach ($stores as $store) {
            //If $store already exist return 'Default Store View'
            if($store->getName() == 'Default Store View') {
                return $store->getName();
            }
        }
        //If default doesn't exist return the first
        foreach ($stores as $store) {
            return $store->getName();
        }
    }

    /**
     * Get customer General group id
     * @return mixed
     */
    public function getGeneralGroupId() {
        $customerGroups = $this->getAllCustomerGroups();
        foreach($customerGroups as $customerGroup) {
            //General group need to exist
            if($customerGroup['label'] == 'General') {
                return $customerGroup['value'];
            }
        }
        //If General group cannot be found return expected id
        return 1;
    }

    /**
     * @return \Magento\Store\Api\Data\StoreInterface[]
     */
    public function getAllStores() {
        if(!isset($this->_allStores)) {
            $this->_allStores = $this->storeManager->getStores(false, false);
        }
        return $this->_allStores;
    }

    /**
     * @return array
     */
    public function getAllStoresCode() {
        $storesCode = [];
        foreach($this->getAllStores() as $store){
            $storesCode[] = $store->getCode();
        }
        return $storesCode;
    }

    /**
     * @return array
     */
    public function getAllStoresId() {
        $storesId = [];
        foreach($this->getAllStores() as $store){
            $storesId[] = $store->getId();
        }
        return $storesId;
    }

    /**
     * @return \Magento\Store\Api\Data\WebsiteInterface[]
     */
    public function getAllWebsites() {
        if(!isset($this->_allWebsites)) {
            $this->_allWebsites = $this->storeManager->getWebsites();
        }
        return $this->_allWebsites;
    }

    public function getAllWebsitesCode() {
        $websitesCode = [];
        foreach($this->getAllWebsites() as $website){
            $websitesCode[] = $website->getCode();
        }
        return $websitesCode;
    }

    public function getAllWebsitesId() {
        $websitesId = [];
        foreach($this->getAllWebsites() as $website){
            $websitesId[] = $website->getId();
        }
        return $websitesId;
    }

    /**
     * @return array
     */
    public function getAllCustomerGroups() {
        if(!isset($this->_allCustomerGroups)) {
            $this->_allCustomerGroups = $this->customerGroupFactory->create()->toOptionArray();
        }
        return $this->_allCustomerGroups;
    }

    /**
     * @param string $storeCode
     * @return bool
     */
    public function getStoreIdByCode(string $storeCode) {
        foreach($this->getAllStores() as $store){
            if($store->getCode() === $storeCode){
                return $store->getId();
            }
        }
        return false;
    }

    /**
     * @param int $storeId
     * @return bool
     */
    public function getStoreCodeById(int $storeId) {
        foreach($this->getAllStores() as $store){
            if((int) $store->getId() === $storeId){
                return $store->getCode();
            }
        }
        return false;
    }

    /**
     * @param string $websiteCode
     * @return bool
     */
    public function getWebsiteIdByCode(string $websiteCode) {
        foreach($this->getAllWebsites() as $website){
            if($website->getCode() === $websiteCode){
                return $website->getId();
            }
        }
        return false;
    }

    /**
     * @param int $websiteId
     * @return bool
     */
    public function getWebsiteCodeById(int $websiteId) {
        foreach($this->getAllWebsites() as $website){
            if((int) $website->getId() === $websiteId){
                return $website->getCode();
            }
        }
        return false;
    }

    /**
     * @param string $country
     * @return \Magento\Store\Api\Data\StoreInterface
     * @throws \Exception
     */
    public function getStoreByCountry(string $country) {
        foreach($this->getAllStores() as $store) {
            $storeCountries = explode(',', $this->getScopeConfig(self::XML_PATH_COUNTRY, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store->getStoreId()));
            if(in_array($country, $storeCountries)) {
                return $store;
            }
        }
        throw new \Exception('Store doesnt exist for country ' . $country);
    }

    /**
     * @param string $country
     * @return int
     * @throws \Exception
     */
    public function getWebsiteByCountry(string $country) {
        return $this->getStoreByCountry($country)->getWebsiteId();
    }

    /**
     * @param string $email
     * @return bool
     */
    public function getEmailIsValid(string $email) {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }
        return true;
    }

    /**
     * @param string $ip
     * @return bool
     */
    public function getIpIsValid(string $ip) {
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return false;
        }
        return true;
    }

    /**
     * Method to have secure utf8 string
     * @param $string
     * @return bool|string
     */
    public function getStringUtf8(string $string) {
        $decoded = utf8_decode($string);
        if (mb_detect_encoding($decoded , 'UTF-8', true) === false) {
            return $string;
        }
        return $decoded;
    }
    
    /**
     * @param array $array
     * @return bool
     */
    public function getArrayIsConsecutive(array $array) {
        return ((int)max($array)-(int)min($array) == (count($array)-1));
    }


}
