<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Helper;

use Magento\Framework\Exception\NoSuchEntityException;

class Customer extends Data implements EntitiesInterface {

    const COMPOSITE_ENTITY_CODE = 'customer_composite';
    const CUSTOMER_ENTITY_CODE = 'customer';

    const XML_PATH_CUSTOMER_BEHAVIOR            = 'dropimport/customer_configuration/behavior';
    const XML_PATH_ATTRIBUTE_IMPORT_CRON           = 'dropimport/customer_configuration/import_cron';

    const XML_PATH_CUSTOMER_IMPORT_METHOD = 'dropimport/customer_configuration/import_method';
    const XML_PATH_CUSTOMER_IMPORT_SPREADSHEET_ID = 'dropimport/customer_configuration/spreadsheet_id';
    const XML_PATH_CUSTOMER_IMPORT_SHEET_TITLE = 'dropimport/customer_configuration/import_sheet_title';
    const XML_PATH_CUSTOMER_IMPORT_PATH_URL = 'dropimport/customer_configuration/import_path_url';
    const XML_PATH_CUSTOMER_IMPORT_GEOLOCALIZE_ADDRESS = 'dropimport/customer_configuration/geolocalize_address';

    private $entityCode;
    /**
     * @var \Magento\Customer\Api\CustomerRepositoryInterface
     */
    private $customerRepository;
    /**
     * @var \Magento\Directory\Model\RegionFactory
     */
    private $regionFactory;

    /**
     * Customer constructor.
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\Config\Storage\WriterInterface $configWriter
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory
     * @param \Magento\Directory\Model\CountryFactory $countryFactory
     * @param \Magento\Customer\Api\CustomerRepositoryInterface $customerRepository
     * @param \Magento\Directory\Model\RegionFactory $regionFactory
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\Config\Storage\WriterInterface $configWriter,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        \Magento\Customer\Api\CustomerRepositoryInterface $customerRepository,
        \Magento\Directory\Model\RegionFactory $regionFactory
    )
    {
        $this->customerRepository = $customerRepository;
        $this->regionFactory = $regionFactory;
        parent::__construct($context, $scopeConfig, $configWriter, $storeManager, $customerGroupFactory, $countryFactory);
    }

    /**
     * Get Behavior
     *
     * @return string
     */
    public function getBehavior()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_BEHAVIOR);
    }

    /**
     * Get Import Cron
     *
     * @return int
     */
    public function getImportCron()
    {
        return $this->getScopeConfig(self::XML_PATH_ATTRIBUTE_IMPORT_CRON);
    }

    /**
     * Get Import Method
     *
     * @return int
     */
    public function getImportMethod()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_IMPORT_METHOD);
    }

    /**
     * Get Spreadsheet Id
     *
     * @return int
     */
    public function getSpreadsheetId()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_IMPORT_SPREADSHEET_ID);
    }

    /**
     * Get Sheet Title
     *
     * @return int
     */
    public function getSheetTitle()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_IMPORT_SHEET_TITLE);
    }

    /**
     * Get Import Path or Url
     *
     * @return int
     */
    public function getImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_IMPORT_PATH_URL);
    }

    /**
     * Get Import Path or Url
     *
     * @return int
     */
    public function getGeolocalizeAddress()
    {
        return $this->getScopeConfig(self::XML_PATH_CUSTOMER_IMPORT_GEOLOCALIZE_ADDRESS);
    }

    /**
     * @param $entityCode
     * @return $this
     * @throws \Exception
     */
    public function setEntityCode($entityCode) {
        if($entityCode != self::CUSTOMER_ENTITY_CODE && $entityCode != self::COMPOSITE_ENTITY_CODE) {
            throw new \Exception('Entity code doesnt exist');
        }
        $this->entityCode = $entityCode;
        return $this;
    }

    public function getEntityCode() {
        if(empty($this->entityCode)) {
            throw new \Exception('Empty entity code');
        }
        return $this->entityCode;
    }

    public function setCustomerEntityCode() {
        return $this->setEntityCode(self::CUSTOMER_ENTITY_CODE);
    }

    public function setCompositeEntityCode() {
        return $this->setEntityCode(self::COMPOSITE_ENTITY_CODE);
    }

    /**
     * Get Entity Code
     *
     * @return string
     */
    public function getCompositeEntityCode()
    {
        return self::COMPOSITE_ENTITY_CODE;
    }

    /**
     * Get Entity Code
     *
     * @return string
     */
    public function getCustomerEntityCode()
    {
        return self::CUSTOMER_ENTITY_CODE;
    }

    /**
     * @param int $groupId
     * @return mixed|null
     */
    public function getGroupId(int $groupId = null) {
        if(empty($groupId)) {
            return $this->getGeneralGroupId();
        }
        return $groupId;
    }

    /**
     * @param string $email
     * @param int $websiteId
     * @return bool
     */
    public function getCustomerExistByEmail(string $email, int $websiteId) {
        try {
            $this->customerRepository->get($email, $websiteId);
        } catch (NoSuchEntityException $e) {
            return false;
        } catch (\Exception $e) {
            return false;
        }
        return true;
    }

    /**
     * @param string $regionCode
     * @param string $countryId
     * @return bool
     */
    public function getRegionExistbyCode(string $regionCode, string $countryId) {
        $region = $this->regionFactory->create()->loadByCode($regionCode, $countryId);
        if($region && $region->getRegionId()) {
            return true;
        }
        return false;
    }

    /**
     * @param string $regionName
     * @param string $countryId
     * @return bool
     */
    public function getRegionExistbyName(string $regionName, string $countryId) {
        $region = $this->regionFactory->create()->loadByName($regionName, $countryId);
        if($region && $region->getRegionId()) {
            return true;
        }
        return false;
    }

    /**
     * @return bool
     */
    public function geocodeModuleExist() {
        if (class_exists(\Limesharp\Stockists\Controller\Adminhtml\Stores\Geocode::class) &&
            class_exists(\Limesharp\Stockists\Model\Geocode::class)) {
            return true;
        }
        return false;
    }

    /**
     * @param $address
     * @return bool
     */
    public function geocodeAddress($address) {
        if(empty($address) || !$this->geocodeModuleExist()) {
            return $address;
        }

        /** @var \Limesharp\Stockists\Model\Geocode $geocodeModel */
        $geocodeModel = $this->getObjectManager()->get('\Limesharp\Stockists\Model\Geocode');
        try {
            return $geocodeModel->completeGeocode($address);
        } catch (\Exception $e) {
            return false;
        }
    }

}
