<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Helper;

class Category extends Data implements EntitiesInterface {
    
    const ENTITY_CODE = 'catalog_category';
    const XML_PATH_CATEGORY_BEHAVIOR            = 'dropimport/category_configuration/behavior';
    const XML_PATH_ATTRIBUTE_IMPORT_CRON           = 'dropimport/category_configuration/import_cron';

    const XML_PATH_CATEGORY_IMPORT_METHOD = 'dropimport/category_configuration/import_method';
    const XML_PATH_CATEGORY_IMPORT_SPREADSHEET_ID = 'dropimport/category_configuration/spreadsheet_id';
    const XML_PATH_CATEGORY_IMPORT_SHEET_TITLE = 'dropimport/category_configuration/import_sheet_title';
    const XML_PATH_CATEGORY_IMPORT_PATH_URL = 'dropimport/category_configuration/import_path_url';

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory
     */
    private $categoryFactory;
    /**
     * @var \Magento\Catalog\Model\CategoryRepository
     */
    private $categoryRepository;

    /**
     * Category constructor.
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\Config\Storage\WriterInterface $configWriter
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory
     * @param \Magento\Directory\Model\CountryFactory $countryFactory
     * @param \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory $categoryFactory
     * @param \Magento\Catalog\Model\CategoryRepository $categoryRepository
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\Config\Storage\WriterInterface $configWriter,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $customerGroupFactory,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory $categoryFactory,
        \Magento\Catalog\Model\CategoryRepository $categoryRepository
    )
    {
        $this->storeManager = $storeManager;
        $this->categoryFactory = $categoryFactory;
        $this->categoryRepository = $categoryRepository;
        parent::__construct($context, $scopeConfig, $configWriter, $storeManager, $customerGroupFactory, $countryFactory);
    }

    /**
     * Get Behavior
     *
     * @return string
     */
    public function getBehavior()
    {
        return $this->getScopeConfig(self::XML_PATH_CATEGORY_BEHAVIOR);
    }

    /**
     * Get Import Cron
     *
     * @return int
     */
    public function getImportCron()
    {
        return $this->getScopeConfig(self::XML_PATH_ATTRIBUTE_IMPORT_CRON);
    }

    /**
     * Get Import Method
     *
     * @return int
     */
    public function getImportMethod()
    {
        return $this->getScopeConfig(self::XML_PATH_CATEGORY_IMPORT_METHOD);
    }

    /**
     * Get Spreadsheet Id
     *
     * @return int
     */
    public function getSpreadsheetId()
    {
        return $this->getScopeConfig(self::XML_PATH_CATEGORY_IMPORT_SPREADSHEET_ID);
    }

    /**
     * Get Sheet Title
     *
     * @return int
     */
    public function getSheetTitle()
    {
        return $this->getScopeConfig(self::XML_PATH_CATEGORY_IMPORT_SHEET_TITLE);
    }

    /**
     * Get Import Path or Url
     *
     * @return int
     */
    public function getImportPathUrl()
    {
        return $this->getScopeConfig(self::XML_PATH_CATEGORY_IMPORT_PATH_URL);
    }

    /**
     * Get Entity Code
     * 
     * @return string
     */
    public function getEntityCode()
    {
        return self::ENTITY_CODE;
    }

    /**
     * @param int $storeId
     * @return mixed
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getRootCategoryId()
    {
        return $this->storeManager->getStore($this->getDefaultStore()->getId())->getRootCategoryId();
    }

    public function getRootCategoryName() {
        $rootCategoryId = $this->getRootCategoryId();
        try {
            return $this->categoryRepository->get($rootCategoryId)->getName();
        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }
    }

    /**
     * @return string
     */
    public function getCategoryPathSeparator() {
        return $this->getScopeConfig(\FireGento\FastSimpleImport\Helper\Config::XML_PATH_CATEGORY_PATH_SEPERATOR);
    }

    public function getCategoryTreeExist(array $categoryTree) {
        $parentId = $this->getRootCategoryId();
        foreach($categoryTree as $categoryName) {
            $parentId = $this->getCategoryExistByNameAndParentId($categoryName, $parentId);
            if(!$parentId) {
                return false;
            }
        }
        return true;
    }

    /**
     * @param string $categoryName
     * @param int $parentId
     * @return int|bool
     */
    public function getCategoryExistByNameAndParentId(string $categoryName, int $parentId) {
        try {
            $collection = $this->categoryFactory->create()
                ->setStore(0)
                ->addAttributeToFilter('name', $categoryName)
                ->addAttributeToFilter('parent_id', $parentId);
            return $collection->getFirstItem()->getId();
        } catch (\Exception $e) {}
        return false;
    }

    /**
     * @param string $categoryName
     * @return int|bool
     */
    public function getCategoryExistByName(string $categoryName) {
        try {
            $collection = $this->categoryFactory->create()
                ->addAttributeToFilter('name', $categoryName)
                ->setPageSize(1);
            return $collection->getFirstItem()->getId();
        } catch (\Exception $e) {}
        return false;
    }

}
