<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Entities\Products;

use Magento\Catalog\Model\Product\Type;
use Magento\CatalogImportExport\Model\Import\Product\Validator\Media;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable as ConfigurableType;

class Swatches extends AbstractProducts
{
    /**
     * @var \Drop\Import\Helper\Product
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Formatter\Product\Swatch
     */
    private $formatter;
    /**
     * @var \Drop\Import\Model\AttributeFactory
     */
    private $attributeFactory;

    /**
     * Products constructor.
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\Product $helper
     * @param \Drop\Import\Model\Checkers\Entities\Product\Swatch $checker
     * @param \Magento\Framework\App\State $state
     * @param \Drop\Import\Model\AttributeFactory $attributeFactory
     * @param \Drop\Import\Model\Formatter\Product\Swatch $formatter
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger,
        \Drop\Import\Helper\Product $helper,
        \Drop\Import\Model\Checkers\Entities\Product\Swatch $checker,
        \Magento\Framework\App\State $state,
        \Drop\Import\Model\AttributeFactory $attributeFactory,
        \Drop\Import\Model\Formatter\Product\Swatch $formatter
    )
    {
        $this->formatter = $formatter;
        $this->helper = $helper;
        $this->attributeFactory = $attributeFactory;
        parent::__construct($objectManagerFactory, $csvProcessor, $spreadsheetsFactory, $logger, $helper, $checker, $state);
    }

    protected function configure()
    {
        $this->setName('drop:import:swatches')
            ->setDescription('Import Swatches Products')
            ->setImportProductType('swatches');
        parent::configure();
    }

    /**
     * @param array $inputData
     * @return array
     */
    protected function getEntities(array $inputData)
    {
        $languages = $this->getTranslationLanguages();
        $productImagesInformation = $this->helper->getProductImagesInformation();

        $products = [];
        $i = 0;
        foreach ($inputData as $key => $row) {

            $products[$i] = [
                'sku'                => $this->formatter->skuFormatter(),
                'store_view_code'    => '',
                'attribute_set_code' => $row['attribute_set_code'],
                'product_type'       => ConfigurableType::TYPE_CODE,
                'product_websites'   => $row['websites'],
                'categories'         => $this->helper->getCategoryHelper()->getRootCategoryName() . '/' . $row['categories'],
                'name'               => $this->formatter->nameFormatter(),
                'description'        => $this->formatter->descriptionFormatter(),
                'price'              => $this->formatter->getProductPrice(),
                'url_key'            => $this->formatter->urlFormatter(),
                'meta_title'         => $this->formatter->metatitleFormatter(),
                'meta_description'   => $this->formatter->metadescriptionFormatter(),
                'visibility'         => $this->formatter->getProductVisibility(),
                'status'             => $this->formatter->getProductStatus(),
                'manage_stock'       => $this->formatter->getProductManageStock(),
                'weight'             => $row['weight']
            ];

            //Images
            foreach($productImagesInformation[$this->formatter->skuFormatter()] as $imageCounter => $imagePath) {
                foreach($this->formatter->imageRoleFormatter($imageCounter) as $imageRole) {
                    if($imageRole == Media::ADDITIONAL_IMAGES && !empty($products[$i][Media::ADDITIONAL_IMAGES])) {
                        $products[$i][$imageRole] .= ',' . $imagePath;
                    } else {
                        $products[$i][$imageRole] = $imagePath;
                    }
                }
            }

            //Custom attributes
            foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code']) as $userDefinedAttribute) {
                $products[$i][$userDefinedAttribute] = $row[$userDefinedAttribute];
            }

            //Translation rows
            $configurableCounter = $i; $i++;
            foreach ($languages as $language) {
                if ($language == $this->helper->getReferenceStoreCode()) {
                    continue;
                }

                $this->formatter->setLanguage($language);

                $products[$i] = [
                    'sku'                => $this->formatter->skuFormatter(),
                    'store_view_code'    => $language,
                    'attribute_set_code' => $row['attribute_set_code'],
                    'product_type'       => ConfigurableType::TYPE_CODE,
                    'name'               => $this->formatter->nameFormatter(),
                    'description'        => $this->formatter->descriptionFormatter(),
                    'url_key'            => $this->formatter->urlFormatter(),
                    'meta_title'         => $this->formatter->metatitleFormatter(),
                    'meta_description'   => $this->formatter->metadescriptionFormatter()
                ];

                // Custom attributes translations
                foreach($this->helper->getUserDefinedAttributes($this->getHeader(), $row['attribute_set_code'], $language) as $userDefinedAttribute) {
                    $products[$i][$userDefinedAttribute] = $row[$userDefinedAttribute];
                }
                $i++;
            }

            $this->formatter->resetLanguage();

            // Simple swatches associated creation
            $attributeModel = $this->attributeFactory->create();
            $superAttributeOptions = $attributeModel->getSuperAttributeOptions($row['attribute_set_code']);
            $swatchesOptions = $attributeModel->getSwatchAttributeOptions($row['attribute_set_code']);
            $swatchVariation = $this->formatter->swatchVariationFormatter();
            foreach ($swatchesOptions as $swatchAttributeCode => $swatchOptions) {
                foreach($swatchOptions as $swatchOption) {
                    if (empty($swatchOption['value']) || empty($swatchOption['label'])) {
                        continue;
                    }

                    if(!in_array($swatchOption['label'], $swatchVariation)) {
                        continue;
                    }

                    // Swatch images
                    $swatchImages = [];
                    foreach($productImagesInformation[$this->formatter->simpleAssociatedSkuFormatter($swatchOption['label'])] as $imageCounter => $imagePath) {
                        foreach($this->formatter->imageRoleFormatter($imageCounter) as $imageRole) {
                            if($imageRole == Media::ADDITIONAL_IMAGES && !empty($swatchImages[Media::ADDITIONAL_IMAGES])) {
                                $swatchImages[$imageRole] .= ',' . $imagePath;
                            } else {
                                $swatchImages[$imageRole] = $imagePath;
                            }
                        }
                    }

                    foreach ($superAttributeOptions as $superAttributeCode => $simpleOptions) {
                        foreach ($simpleOptions as $simpleOption) {
                            if (empty($simpleOption['value']) || empty($simpleOption['label'])) {
                                continue;
                            }

                            $products[$configurableCounter]['configurable_variations'][] = [
                                'sku'               => $this->formatter->simpleAssociatedSkuFormatter($swatchOption['label'] . '-' . $simpleOption['label']),
                                $superAttributeCode => $simpleOption['label'],
                                $swatchAttributeCode => $swatchOption['label'],
                            ];

                            $products[$i] = [
                                'sku'                => $this->formatter->simpleAssociatedSkuFormatter($swatchOption['label'] . '-' . $simpleOption['label']),
                                'store_view_code'    => '',
                                'attribute_set_code' => $row['attribute_set_code'],
                                'product_type'       => Type::TYPE_SIMPLE,
                                'product_websites'   => $row['websites'],
                                'name'               => $this->formatter->simpleAssociatedNameFormatter(' - ' . $swatchOption['label'] . ': ' . $simpleOption['label']),
                                $superAttributeCode  => $simpleOption['label'],
                                $swatchAttributeCode  => $swatchOption['label'],
                                'categories'         => $this->helper->getCategoryHelper()->getRootCategoryName() . '/' . $row['categories'],
                                'price'              => $this->formatter->getProductPrice(),
                                'weight'             => $row['weight'],
                                'visibility'         => $this->formatter->getProductVisibility(),
                                'status'             => $this->formatter->getProductStatus(),
                                'url_key'            => $this->formatter->simpleAssociatedUrlFormatter(null, $swatchOption['label'] . '-' . $simpleOption['label'])
                            ];

                            $products[$i] = array_merge($products[$i], $swatchImages);

                            $i++;
                        }
                    }
                }
            }
        }

        return $products;
    }

}
