<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Entities;

use \Drop\Import\Console\Command\Import\AbstractImport;
use Magento\Framework\App\State;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Console\Cli;

abstract class AbstractImportEntities extends AbstractImport
{
    /**
     * @var ObjectManagerFactory
     */
    private $objectManagerFactory;
    /**
     * @var Magento\Framework\App\State
     */
    private $state;
    /**
     * @var \Drop\Import\Helper\EntitiesInterface
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Checkers\CheckerInterface
     */
    private $checker;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\EntitiesInterface $helper
     * @param \Drop\Import\Model\Checkers\CheckerInterface $checker
     * @param State $state
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger,
        \Drop\Import\Helper\EntitiesInterface $helper,
        \Drop\Import\Model\Checkers\CheckerInterface $checker,
        \Magento\Framework\App\State $state
    ) {
        $this->objectManagerFactory = $objectManagerFactory;
        $this->state = $state;
        $this->helper = $helper;
        $this->checker = $checker;

        parent::__construct($objectManagerFactory, $csvProcessor, $spreadsheetsFactory, $logger);
    }

    /**
     * @param \Symfony\Component\Console\Input\InputInterface $input
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @return int|void|null
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $this->state->getAreaCode();
        } catch (\Exception $e) {
            $this->state->setAreaCode('adminhtml');
        }

        $this->setOutput($output);

        // Set option argument variable
        $definitionOptions = $this->getDefinition()->getOptions();
        $options = [];
        foreach($definitionOptions as $option) {
            $optionValue = $input->getOption($option->getName());
            if($optionValue) {
                $options[$option->getName()] = $optionValue;
            }
        }
        $this->setOptionsArgument($options);

        $this->import();
    }

    /**
     * @return int
     * @throws \Exception
     */
    public function import() {

        // Before scripts
        $this->log('Before Import scripts...' , 'comment');
        $this->beforeImport();

        // Read data
        $importMethod = $this->helper->getImportMethod();
        if ($importMethod == \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_FILE_PATH_URL) {
            $inputData = $this->readCSV(BP . '/' . $this->helper->getImportPathUrl());
        } elseif($importMethod == \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_GOOGLE_DRIVE) {
            $inputData = $this->readSpreadsheet($this->helper->getSpreadsheetId(), $this->helper->getSheetTitle());
        }

        if(empty($inputData)) {
            $this->log('Empty input data. Exiting', 'error');
            return Cli::RETURN_FAILURE;
        }

        // Import properties and model definition
        /** @var \FireGento\FastSimpleImport\Model\Importer $importerModel */
        $this->setObjectManager();
        $importerModel = $this->getObjectManager()->create('FireGento\FastSimpleImport\Model\Importer');
        $importerModel->setBehavior($this->helper->getBehavior());
        $importerModel->setEntityCode($this->helper->getEntityCode());
        $adapterFactory = $this->getObjectManager()->create('FireGento\FastSimpleImport\Model\Adapters\NestedArrayAdapterFactory');
        $importerModel->setImportAdapterFactory($adapterFactory);

        $this->log('Configuration: ', 'comment');
        $this->log('Behavior: ' . $this->helper->getBehavior());
        $this->log('EntityCode: ' . $this->helper->getEntityCode());
        $dryRunModeEnabledLabel = 'No';
        if($this->helper->getDryRunIsEnabled()) {
            $dryRunModeEnabledLabel = '<success>Yes</success>';
        }
        $this->log('DryRunMode: ' . $dryRunModeEnabledLabel);
        $testModeEnabledLabel = 'No';
        if($this->helper->getTestModeIsEnabled()) {
            $testModeEnabledLabel = '<success>Yes</success>';
        }
        $this->log('TestMode: ' . $testModeEnabledLabel);

        // Data validation
        $this->log('Start data check...', 'comment');
        if(!$this->checker->setTranslationLanguage($this->getTranslationLanguages())->setHeader($this->getHeader())->validateData($inputData)->getValidationResult()) {
            $this->log('The data check has reported the following errors');
            $this->log($this->checker->getErrorMessages(), 'error');
            if($this->getIsCron() && !empty($this->helper->getEmailReport())) {
                $this->sendEmailLog($this->getEmailSubject(), $this->helper->getFromSender(), $this->helper->getEmailReport());
            }
            return Cli::RETURN_FAILURE;
        } else {
            $this->log('The data check is completed. All data appears to be correct.');
        }

        $warningMessages = $this->checker->getWarningMessages();
        if(!empty($warningMessages)) {
            $this->log($warningMessages, 'comment');
        }

        try {
            if(!$this->helper->getDryRunIsEnabled()) {
                $this->log('Import started...', 'comment');
            }

            $data = $this->getEntities($inputData);
            if(!$this->helper->getArrayIsConsecutive(array_keys($data))) {
                $this->log('Entities array is not consecutive ' . implode(',', array_keys($data)) . '. There are errors in the script. Import cannot be performed.', 'error');
                return Cli::RETURN_FAILURE;
            }

            if($this->helper->getTestModeIsEnabled()) {
                $this->log('Test mode is enabled. Limit import to ' . self::TEST_MODE_NUMBER . ' elements.', 'question');
                $data = array_slice($data, 0, self::TEST_MODE_NUMBER);
            }

            if(!$this->helper->getDryRunIsEnabled()) {
                $importerModel->processImport($data);
            } else {
                $this->log('Logging input Data...', 'comment');
                $this->log($data);
            }
        } catch (\Exception $e) {
            $this->log($e->getMessage(), 'error');
        }

        $this->log($importerModel->getLogTrace());
        $errorMessages = $importerModel->getErrorMessages();
        if(!empty($errorMessages)) {
            $this->log($importerModel->getErrorMessages(), 'error');
        }

        // After scripts
        $this->log('After Import scripts...', 'comment');
        $this->afterImport();

        if(!$this->helper->getDryRunIsEnabled()) {
            $this->log('Import finished.', 'comment');
        } else {
            $this->log('Logging finished.', 'comment');
        }

        if(($this->helper->getSendEmailReport() || $this->getIsCron()) && !empty($this->helper->getEmailReport())) {
            $this->sendEmailLog($this->getEmailSubject(), $this->helper->getFromSender(), $this->helper->getEmailReport());
        }
    }

    private function getEmailSubject() {
        return $this->helper->getStoreName() . ' ' . $this->helper->getEntityCode() . ' Import log';
    }

    /**
     * @param array $inputData
     * @return array
     */
    abstract protected function getEntities(array $inputData);

}
