<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Command;

use Symfony\Component\Console\Output\OutputInterface;
use \Magento\Framework\Exception\AlreadyExistsException;

class Urlrewrite extends AbstractImportCommand
{
    /**
     * @var Magento\UrlRewrite\Model\UrlRewrite
     */
    private $urlRewrite;
    /**
     * @var Magento\UrlRewrite\Helper\UrlRewrite
     */
    private $urlRewriteHelper;

    /**
     * @var \Drop\Import\Helper\Data
     */
    private $helper;

    /**
     * Urlrewrite constructor.
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\UrlRewrite $helper
     * @param \Drop\Import\Model\Checkers\Command\UrlRewrite $checker
     * @param \Magento\Framework\App\State $state
     * @param \Magento\UrlRewrite\Model\UrlRewrite $urlRewrite
     * @param \Magento\UrlRewrite\Helper\UrlRewrite $urlRewriteHelper
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger,
        \Drop\Import\Helper\UrlRewrite $helper,
        \Drop\Import\Model\Checkers\Command\UrlRewrite $checker,
        \Magento\Framework\App\State $state,
        \Magento\UrlRewrite\Model\UrlRewrite $urlRewrite,
        \Magento\UrlRewrite\Helper\UrlRewrite $urlRewriteHelper
    )
    {
        $this->urlRewrite = $urlRewrite;
        $this->urlRewriteHelper = $urlRewriteHelper;
        $this->helper = $helper;
        parent::__construct($objectManagerFactory, $csvProcessor, $spreadsheetsFactory, $logger, $helper, $checker, $state);
    }

    protected function configure()
    {
        $this->setName('drop:import:url_rewrite')
            ->setDescription('Import Url Rewrite');
    }

    /**
     * @param array $inputData
     * @param OutputInterface $output
     * @return void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function processImport(array $inputData) {
        $successCounter = 0; $errorCounter = 0; $testCounter = 0;
        foreach($inputData as $key => $row) {
            $this->urlRewrite->load(0);
            $this->urlRewriteHelper->validateRequestPath($this->helper->getCleanedRequestPath($row['request_path']));
            $this->urlRewrite->setEntityType('custom')
                ->setRequestPath($this->helper->getCleanedRequestPath($row['request_path']))
                ->setTargetPath($this->helper->getCleanedTargetPath($row['target_path']))
                ->setRedirectType($row['redirect_type'])
                ->setStoreId($row['store_id'])
                ->setDescription("Imported Urls");

            try {
                if(!$this->helper->getDryRunIsEnabled()) {
                    $this->urlRewrite->save();
                }
                unset($this->urlRewrite);
                $successCounter++;
            } catch (AlreadyExistsException $e) {
                $this->log($row['request_path'] . " already exist.");
                continue;
            } catch (\Exception $e) {
                $this->log($e->getMessage(), 'error');
                $errorCounter++;
                continue;
            }

            $testCounter++;
            if($this->helper->getTestModeIsEnabled() && $testCounter >= self::TEST_MODE_NUMBER) {
                break;
            }
        }

        $this->log("Imported successfully " . $successCounter . ' url rewrite', 'success');
        if($errorCounter) {
            $this->log("Not imported " . $errorCounter . ' url rewrite. Are more checks needed?', 'error');
        }
    }

}
