<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Command;

class Subscriber extends AbstractImportCommand
{
    /**
     * @var \Drop\Import\Helper\CommandInterface
     */
    private $helper;
    /**
     * @var Magento\Newsletter\Model\SubscriberFactory
     */
    private $subscriberFactory;

    protected $subscriberNotificationConfiguration;

    /**
     * Subscriber constructor.
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\Subscriber $helper
     * @param \Drop\Import\Model\Checkers\Command\Subscriber $checker
     * @param \Magento\Framework\App\State $state
     * @param \Magento\Newsletter\Model\SubscriberFactory $subscriberFactory
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger,
        \Drop\Import\Helper\Subscriber $helper,
        \Drop\Import\Model\Checkers\Command\Subscriber $checker,
        \Magento\Framework\App\State $state,
        \Magento\Newsletter\Model\SubscriberFactory $subscriberFactory
    )
    {
        $this->helper = $helper;
        $this->subscriberFactory = $subscriberFactory;
        parent::__construct($objectManagerFactory, $csvProcessor, $spreadsheetsFactory, $logger, $helper, $checker, $state);
    }

    protected function configure()
    {
        $this->setName('drop:import:newsletter_subscriber')
            ->setDescription('Import Newsletter Subscriber');
    }

    public function beforeImport() {
        $this->subscriberNotificationConfiguration = $this->helper->getSubscriberConfirmation();
        $this->helper->setSubscriberConfirmation(0);
        $this->log('Unsetted Subscriber Confirmation configuration');
        parent::beforeImport();
    }

    /**
     * @param array $inputData
     * @return void
     */
    public function processImport(array $inputData) {
        $successCounter = 0; $errorCounter = 0; $testCounter = 0;
        foreach($inputData as $key => $row) {
            $subscriberModel = $this->subscriberFactory->create();
            $subscriber = $subscriberModel->loadByEmail($row['email']);
            if($subscriber->isSubscribed()) {
                $this->log($row['email'] . " is already subscribed");
                continue;
            }

            try {
                if(!$this->helper->getDryRunIsEnabled()) {
                    $subscriberModel->subscribe($row['email']);
                    $successCounter++;
                }
                $this->log('Subscribed: ' . $row['email']);
            } catch (\Exception $e) {
                $this->log($e->getMessage(), 'error');
                $errorCounter++;
            }

            $testCounter++;
            if($this->helper->getTestModeIsEnabled() && $testCounter >= self::TEST_MODE_NUMBER) {
                break;
            }
        }

        $this->log("Imported successfully " . $successCounter . ' newsletter subscriber', 'success');
        if($errorCounter) {
            $this->log("Not imported " . $errorCounter . ' newsletter subscriber. Are more checks needed?', 'error');
        }

    }

    public function afterImport() {
        $this->helper->setSubscriberConfirmation($this->subscriberNotificationConfiguration);
        $this->log('Resetted Subscriber Confirmation to previous configuration');
        parent::afterImport();
    }

}
