<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import\Command;

use \Drop\Import\Console\Command\Import\AbstractImport;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Console\Cli;

abstract class AbstractImportCommand extends AbstractImport
{
    /**
     * @var \Magento\Framework\App\State
     */
    private $state;
    /**
     * @var \Drop\Import\Helper\CommandInterface
     */
    private $helper;
    /**
     * @var \Drop\Import\Model\Checkers\CheckerInterface
     */
    private $checker;

    /**
     * AbstractImportCommand constructor.
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     * @param \Drop\Import\Helper\CommandInterface $helper
     * @param \Drop\Import\Model\Checkers\CheckerInterface $checker
     * @param \Magento\Framework\App\State $state
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger,
        \Drop\Import\Helper\CommandInterface $helper,
        \Drop\Import\Model\Checkers\CheckerInterface $checker,
        \Magento\Framework\App\State $state
    ) {
        $this->state = $state;
        $this->helper = $helper;
        $this->checker = $checker;
        parent::__construct($objectManagerFactory, $csvProcessor, $spreadsheetsFactory, $logger);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $this->state->getAreaCode();
        } catch (\Exception $e) {
            $this->state->setAreaCode('adminhtml');
        }

        $this->setOutput($output);

        // Set option argument variable
        $definitionOptions = $this->getDefinition()->getOptions();
        $options = [];
        foreach($definitionOptions as $option) {
            $optionValue = $input->getOption($option->getName());
            if($optionValue) {
                $options[$option->getName()] = $optionValue;
            }
        }
        $this->setOptionsArgument($options);

        $this->import();
    }

    public function import() {
        // Read data
        $importMethod = $this->helper->getImportMethod();
        if ($importMethod == \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_FILE_PATH_URL) {
            $inputData = $this->readCSV(BP . '/' . $this->helper->getImportPathUrl());
        } elseif($importMethod == \Drop\Import\Model\Config\Source\ImportMethod::IMPORT_METHOD_GOOGLE_DRIVE) {
            $inputData = $this->readSpreadsheet($this->helper->getSpreadsheetId(), $this->helper->getSheetTitle());
        }

        if(empty($inputData)) {
            $this->log('Empty input data. Exiting', 'error');
            return Cli::RETURN_FAILURE;
        }

        // Before scripts
        $this->log('Before Import scripts...' , 'comment');
        $this->beforeImport();

        $this->log('Configuration: ', 'comment');
        $this->log('EntityCode: ' . $this->helper->getScriptNameForEmailLog());
        $dryRunModeEnabledLabel = 'No';
        if($this->helper->getDryRunIsEnabled()) {
            $dryRunModeEnabledLabel = '<success>Yes</success>';
        }
        $this->log('DryRunMode: ' . $dryRunModeEnabledLabel);
        $testModeEnabledLabel = 'No';
        if($this->helper->getTestModeIsEnabled()) {
            $testModeEnabledLabel = '<success>Yes</success>';
        }
        $this->log('TestMode: ' . $testModeEnabledLabel);

        if($this->helper->getTestModeIsEnabled()) {
            $this->log('The test mode is enabled but for import commands without entities, it is responsibility of the individual script to honor the test mode. I hope it was well programmed :)', 'question');
        }
        if($this->helper->getDryRunIsEnabled()) {
            $this->log('The dry run mode is enabled but for import commands without entities, it is responsibility of the individual script to honor the dry run mode. I hope it was well programmed :)', 'question');
        }


        // Data validation
        $this->log('Start data check...', 'comment');
        if(!$this->checker->validateData($inputData)->getValidationResult()) {
            $this->log('The data check has reported the following errors');
            $this->log($this->checker->getErrorMessages(), 'error');
            if($this->getIsCron() && !empty($this->helper->getEmailReport())) {
                $this->sendEmailLog($this->getEmailSubject(), $this->helper->getFromSender(), $this->helper->getEmailReport());
            }
            return Cli::RETURN_FAILURE;
        } else {
            $this->log('The data check is completed. All data appears to be correct.');
        }

        $warningMessages = $this->checker->getWarningMessages();
        if(!empty($warningMessages)) {
            $this->log($warningMessages, 'comment');
        }

        // Start!
        try {
            if(!$this->helper->getDryRunIsEnabled()) {
                $this->log('Import started...', 'comment');
            } else {
                $this->log('Logging input Data...', 'comment');
            }

            $result = $this->processImport($inputData);

            if(!empty($result)) {
                $this->log($result, 'success');
            }
        } catch (\Exception $e) {
            $this->log($e->getMessage(), 'error');
        }

        // After scripts
        $this->log('After Import scripts...', 'comment');
        $this->afterImport();

        if(!$this->helper->getDryRunIsEnabled()) {
            $this->log('Import finished.', 'comment');
        } else {
            $this->log('Logging finished.', 'comment');
        }

        if($this->helper->getSendEmailReport() && !empty($this->helper->getEmailReport())) {
            $this->sendEmailLog($this->getEmailSubject(), $this->helper->getFromSender(), $this->helper->getEmailReport());
        }
    }

    private function getEmailSubject() {
        return $this->helper->getStoreName() . ' ' . $this->helper->getScriptNameForEmailLog() . ' Import log';
    }

    /**
     * @param array $inputData
     * @return array
     */
    abstract protected function processImport(array $inputData);


}
