<?php
/**
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Import\Console\Command\Import;

use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManager;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Formatter\OutputFormatterStyle;
use Symfony\Component\Console\Output\OutputInterface;

abstract class AbstractImport extends Command
{
    /**
     * @var \Magento\Framework\File\Csv
     */
    protected $csvProcessor;
    /**
     * @var \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory
     */
    protected $spreadsheetsFactory;
    /**
     * @var \Symfony\Component\Console\Output\OutputInterface
     */
    protected $output;

    public $log = '';
    public $header = [];
    public $translationLanguages = [];
    public $optionsArgument = [];
    private $defaultCacheTags = [
        'config',
        'layout',
        'block_html',
        'collections',
        'reflection',
        'db_ddl',
        'eav',
        'config_integration',
        'config_integration_api',
        'full_page',
        'translate',
        'config_webservice'
    ];
    private $defaultIndex = [
        'design_config_grid',
        'customer_grid',
        'catalog_product_flat',
        'catalog_category_flat',
        'catalog_category_product',
        'catalog_product_category',
        'catalogrule_rule',
        'catalog_product_attribute',
        'cataloginventory_stock',
        'catalogsearch_fulltext',
        'catalog_product_price',
        'catalogrule_product'
    ];
//    private $replaceCharacter = [
//        "’", "'"
//    ];

    const TEST_MODE_NUMBER = 5;
    /**
     * @var \Drop\Import\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Import\Console\Command\Import\bjectManagerFactory
     */
    private $objectManagerFactory;
    /**
     * @var \Drop\Import\Console\Command\Import\bjectManager
     */
    private $objectManager;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory
     * @param \Magento\Framework\File\Csv $csvProcessor
     * @param \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory
     * @param \Drop\Import\Logger\Logger $logger
     */
    public function __construct(
        \Magento\Framework\App\ObjectManagerFactory $objectManagerFactory,
        \Magento\Framework\File\Csv $csvProcessor,
        \Drop\GoogleApiAdapter\Model\Service\SpreadsheetsFactory $spreadsheetsFactory,
        \Drop\Import\Logger\Logger $logger
    ) {
        $this->objectManagerFactory = $objectManagerFactory;
        $this->csvProcessor = $csvProcessor;
        $this->spreadsheetsFactory = $spreadsheetsFactory;
        $this->logger = $logger;

        parent::__construct();
    }

    /**
     * Read CSV file
     * @param string $fileName
     * @return array
     * @throws \Exception
     */
    protected function readCSV($fileName)
    {
        $data = $this->csvProcessor->getData($fileName);
        return $this->formatInputData($data);
    }

    /**
     * Read CSV file
     * @param $spreadSheetId
     * @param $sheetTitle
     * @return array
     * @throws \Exception
     */
    protected function readSpreadsheet($spreadSheetId, $sheetTitle)
    {
        if(empty($spreadSheetId)) {
            throw new \Exception('Empty spreadsheet id configuration');
        }
        if(empty($sheetTitle)) {
            throw new \Exception('Empty sheet title configuration');
        }

        /** @var \Drop\GoogleApiAdapter\Model\Service\Spreadsheets $spreadSheet */
        $spreadSheet = $this->spreadsheetsFactory->create();
        $data = $spreadSheet->getSingleSheetSpreadsheetValues($spreadSheetId, $sheetTitle);
        return $this->formatInputData($data);
    }

    /**
     * @param $data
     * @return array
     */
    protected function formatInputData($data) {
        $this->setHeader(array_shift($data));

        $i = 0;
        $formattedData = [];
        foreach ($data as $row) {
            foreach ($row as $key => $value) {
                if(!isset($this->header[$key])){continue;}
                $formattedData[$i][$this->header[$key]] = $value;
            }
            $i++;
        }
        return $formattedData;
    }

    /**
     * @param array $optionsArgument
     * @return void
     */
    public function setOptionsArgument(array $optionsArgument)
    {
        $this->optionsArgument = $optionsArgument;
    }

    /**
     * @return array
     */
    public function getOptionsArgument()
    {
        return $this->optionsArgument;
    }

    /**
     * @eturn array
     */
    public function getHeader()
    {
        return $this->header;
    }

    /**
     * @eturn array
     * @param $header
     * @return
     */
    public function setHeader(array $header)
    {
        $this->header = $header;
        return $this;
    }

    public function getObjectManager()
    {
        if(empty($this->objectManager)) {
            $this->setObjectManager();
        }
        return $this->objectManager;
    }

    public function setObjectManager()
    {
        $this->objectManager = $this->objectManagerFactory->create($this->getServerParams());

        /** @var \Magento\Framework\App\State $appState */
        $appState = $this->objectManager->get(\Magento\Framework\App\State::class);
        $appState->setAreaCode(\Magento\Backend\App\Area\FrontNameResolver::AREA_CODE);
        $configLoader = $this->objectManager->get(\Magento\Framework\ObjectManager\ConfigLoaderInterface::class);
        $this->objectManager->configure($configLoader->load(\Magento\Backend\App\Area\FrontNameResolver::AREA_CODE));
        return $this;
    }

    /**
     * Clean invalidated caches after import
     * Reindex invalidated index after import
     */
    public function afterImport(){
        $doSomething = false;
        $invalidatedIndexTags = $this->getInvalidatedIndex();
        if(!empty($invalidatedIndexTags)) {
            $this->reindex($invalidatedIndexTags);
            $doSomething = true;
        }
        $invalidatedCacheTags = $this->getInvalidatedCache();
        if(!empty($invalidatedCacheTags)) {
            $this->clearCache($invalidatedCacheTags);
            $doSomething = true;
        }

        if(!$doSomething) {
            $this->log('Nothing to do after import');
        }
        return $this;
    }

    /**
     * Clean invalidated caches before import
     * Reindex invalidated index before import
     */
    public function beforeImport(){
        $doSomething = false;
        $invalidatedIndexTags = $this->getInvalidatedIndex();
        if(!empty($invalidatedIndexTags)) {
            $this->reindex($invalidatedIndexTags);
            $doSomething = true;
        }
        $invalidatedCacheTags = $this->getInvalidatedCache();
        if(!empty($invalidatedCacheTags)) {
            $this->clearCache($invalidatedCacheTags);
            $doSomething = true;
        }

        if(!$doSomething) {
            $this->log('Nothing to do before import');
        }
        return $this;
    }

    /**
     * @return $this
     */
    protected function setTranslationLanguages() {
        $this->translationLanguages = [];
        if(count($this->header)) {
            /** @var \Drop\Import\Helper\Data $helper */
            $helper = $this->getObjectManager()->get('\Drop\Import\Helper\Data');

            foreach ($this->header as $key) {
                $storeCode = $helper->getStoreCodeFromFieldName($key);
                if (!empty($storeCode) && !in_array($storeCode, $this->translationLanguages)) {
                    $this->translationLanguages[] = $storeCode;
                }
            }
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getTranslationLanguages() {
        if(empty($this->translationLanguages)) {
            $this->setTranslationLanguages();
        }
        return $this->translationLanguages;
    }

    /**
     * Get Resource Connection
     *
     * @return \Magento\Framework\App\ResourceConnection
     */
    public function getResources() {
        return $this->getObjectManager()->get(\Magento\Framework\App\ResourceConnection::class);
    }

    /**
     * @param array $cacheTags
     * @return \Drop\Import\Console\Command\Import\AbstractImport
     */
    public function clearCache(array $cacheTags) {
        $_cacheTypeList = $this->getObjectManager()->get('Magento\Framework\App\Cache\TypeListInterface');
        $_cacheFrontendPool = $this->getObjectManager()->get('Magento\Framework\App\Cache\Frontend\Pool');

        try {
            $cleanedCache = false;
            foreach ($cacheTags as $cacheTag) {
                if (in_array($cacheTag, $this->defaultCacheTags)) {
                    $this->log('Cleaning ' . $cacheTag . ' cache tag...');
                    $_cacheTypeList->cleanType($cacheTag);
                    $cleanedCache = true;
                }
            }

            if($cleanedCache) {
                foreach ($_cacheFrontendPool as $cacheFrontend) {
                    $cacheFrontend->getBackend()->clean();
                }
            }
        } catch (\Exception $e) {
            $this->getOutput()->writeln($e->getMessage());
        }

        $this->log('Cache cleaned', 'success');
        return $this;
    }

    /**
     * @return $this
     */
    public function clearCacheAll() {
        $this->log('Cleaning all caches...');
        $_cacheTypeList = $this->getObjectManager()->get('Magento\Framework\App\Cache\TypeListInterface');
        $_cacheFrontendPool = $this->getObjectManager()->get('Magento\Framework\App\Cache\Frontend\Pool');

        foreach ($this->defaultCacheTags as $type) {
            $_cacheTypeList->cleanType($type);
        }
        foreach ($_cacheFrontendPool as $cacheFrontend) {
            $cacheFrontend->getBackend()->clean();
        }
        $this->log('All Caches cleaned');
        return $this;
    }

    /**
     * @param array $indexTags
     * @return \Drop\Import\Console\Command\Import\AbstractImport
     */
    public function reindex(array $indexTags) {
        $indexerFactory = $this->getObjectManager()->get('\Magento\Indexer\Model\IndexerFactory');

        try {
            foreach ($indexTags as $indexTag) {
                if (in_array($indexTag, $this->defaultIndex)) {
                    $this->log('Reindexing ' . $indexTag . ' tag...');
                    $indexerFactory->create()->load($indexTag)->reindexAll();
                }
            }
        } catch (\Exception $e) {
            $this->getOutput()->writeln($e->getMessage());
        }

        $this->log('Reindexed', 'success');
        return $this;
    }

    /**
     * @return $this
     */
    public function reindexAll(){
        $indexer = $this->getObjectManager()->get('\Magento\Indexer\Model\IndexerFactory')->create();
        $this->log('Reindexing all...');
        foreach ($this->indexTags as $indexerId) {
            try {
                $indexer->load($indexerId)->reindexAll();
            } catch (\Exception $e) {
                $this->getOutput()->writeln($e->getMessage());
            }
        }

        $this->log('Reindexed all', 'success');
        return $this;
    }

    /**
     * @return array
     */
    public function getInvalidatedCache() {
        $_cacheTypeList = $this->getObjectManager()->get('Magento\Framework\App\Cache\TypeListInterface');
        $invalidCaches = $_cacheTypeList->getInvalidated();
        $invalidCacheTags = [];
        foreach($invalidCaches as $key => $value) {
            $invalidCacheTags[] = $key;
        }
        return $invalidCacheTags;
    }

    /**
     * @return array
     */
    public function getInvalidatedIndex() {
        // Make sure the indexes are ready
        $indexerFactory = $this->getObjectManager()->create('\Magento\Indexer\Model\IndexerFactory');
        $indexerCollection = $this->getObjectManager()->get('\Magento\Indexer\Model\Indexer\CollectionFactory')->create();
        $invalidIndexTags = [];
        foreach ($indexerCollection->getAllIds() as $indexId){
            $index = $indexerFactory->create()->load($indexId);
            if($index->getStatus() == 'invalid') {
                $invalidIndexTags[] = $indexId;
            }
        }
        return $invalidIndexTags;
    }

    /**
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @return \Drop\Import\Console\Command\Import\AbstractImport
     */
    protected function setOutput(OutputInterface $output) {
        $this->output = $output;
        $this->setCustomStyle();
        return $this;
    }

    /**
     * Set custom tag style to format output
     */
    protected function setCustomStyle() {
        $this->getOutput()->getFormatter()
            ->setStyle('success', new OutputFormatterStyle('white', 'green'));
    }

    /**
     * @return \Symfony\Component\Console\Output\OutputInterface
     */
    protected function getOutput() {
        return $this->output;
    }

    protected function getIsCron() {
        if(!isset($_SERVER['SSH_CLIENT'])) {
            return true;
        }
        return false;
    }

    /**
     * @param $message
     * @param string|null $level
     * @return $this
     */
    protected function log($message, string $level = null) {
        if(is_array($message)) {
            $stringifyArray = '';
            foreach($message as $row) {
                $stringifyArray .= $row . '<br/>';
            }
            $message = $stringifyArray;
        }

        switch ($level) {
            case 'info':
                $this->logger->info($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln(PHP_EOL . '<info>' . $message . '</info>');
                }
                $this->log .= '<br/><span style="background-color:yellow">' . $this->cliToHtmlFormat($message) . '</span><br/>';
                break;
            case 'comment':
                $this->logger->warning($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln(PHP_EOL . '<comment>' . $message . '</comment>');
                }
                $this->log .= '<br/><span style="background-color:yellow">' . $this->cliToHtmlFormat($message) . '</span><br/>';
                break;
            case 'question':
                $this->logger->warning($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln(PHP_EOL . '<question>' . $message . '</question>');
                }
                $this->log .= '<br/><p style="background-color:blue;color:white;text-align:center;padding:10px;">' . $this->cliToHtmlFormat($message) . '</p><br/>';
                break;
            case 'error':
                $this->logger->error($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln(PHP_EOL . '<error>' . $message . '</error>');
                }
                $this->log .= '<br/><p style="background-color:red;color:white;text-align:center;padding:10px;">' . $this->cliToHtmlFormat($message) . '</p><br/>';
                break;
            case 'success':
                $this->logger->info($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln(PHP_EOL . '<success>' . $message . '</success>');
                }
                $this->log .= '<br/><p style="background-color:green;color:white;text-align:center;padding:10px;">' . $this->cliToHtmlFormat($message) . '</p><br/>';
                break;
            default:
                $this->logger->info($message);
                if(!$this->getIsCron()) {
                    $this->getOutput()->writeln($message);
                }
                $this->log .= '<small>' . $this->cliToHtmlFormat($message) . '</small><br/>';
                break;
        }

        return $this;
    }

    /**
     * @param $subject
     * @param $from
     * @param $to
     * @return bool
     */
    protected function sendEmailLog($subject, $from, $to) {
        $transportBuilder = $this->getObjectManager()->get('Magento\Framework\Mail\Template\TransportBuilder');
        $transport = $transportBuilder
            ->setTemplateIdentifier('drop_import_log_email_template')
            ->setTemplateOptions(['area' => \Magento\Framework\App\Area::AREA_FRONTEND, 'store' => \Magento\Store\Model\Store::DEFAULT_STORE_ID])
            ->setTemplateVars([
                'title' => $subject,
                'message' => $this->log
            ])
            ->setFromByScope($from)
            ->addTo($to)
            ->getTransport();

        try {
            $transport->sendMessage();
            $this->log('Email log was sent.', 'comment');
            return true;
        } catch (\Exception $e) {
            $this->log($e->getMessage(), 'error');
        }
        return false;
    }

    /**
     * @param string $string
     * @return string $string
     */
    protected function cliToHtmlFormat(string $string) {
        return str_replace(PHP_EOL, '<br/>', $string);
    }

    /**
     * @return array
     */
    protected function getServerParams() {
        $serverParams = $_SERVER;
        $serverParams[StoreManager::PARAM_RUN_CODE] = 'admin';
        $serverParams[Store::CUSTOM_ENTRY_POINT_PARAM] = true;
        return $serverParams;
    }

}
