<?php

namespace Drop\Gsped\Model\Sales;

class ShipmentRepositoryExtension {

    /**
     * @var \Drop\Gsped\Logger\Logger
     */
    protected $logger;

    /**
     * @var \Magento\Sales\Model\Order\Shipment\ItemFactory
     */
    protected $shipmentItemFactory;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resourceConnection;

    public function __construct(
        \Drop\Gsped\Logger\Logger $logger,
        \Magento\Sales\Model\Order\Shipment\ItemFactory $shipmentItemFactory,
        \Magento\Framework\App\ResourceConnection $resourceConnection
    ) {
        $this->logger = $logger;
        $this->shipmentItemFactory = $shipmentItemFactory;
        $this->resourceConnection = $resourceConnection;
    }

    /**
     *
     * An issue in Magento2 causes an order's status to remain unchanged when the order is shipped via the API
     * Shipment creation does not update the qty_shipped for an order's sales_order_items, preventing the order state and status from changing.
     *
     * @param \Magento\Checkout\Model\ShipmentRepository $subject
     * @param \Magento\Sales\Model\Order\Shipment
     */
    public function afterSave(
        \Magento\Sales\Model\Order\ShipmentRepository $subject,
        \Magento\Sales\Model\Order\Shipment $result
    ) {
        $shipmentId = $result->getId();
        $shipment = $subject->get($shipmentId);
        $order = $result->getOrder()->load($result->getOrder()->getId());

        $shippingItemsData = [];
        foreach($order->getAllItems() as $orderItem) {
            //Creating an array of order item data to add missing data to shipping items and create missing items
            $shippingItemsData[$orderItem->getId()] = $orderItem->getData();

            try {
                //Set qty shipped on invoice creation starting from qty invoiced (we all ship all items together)
                $orderItem->setQtyShipped((int) $orderItem->getQtyInvoiced())->save();
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }
        
        foreach ($result->getAllItems() as $shippingItem) {
            try {
                //If shipping item exist force update with order item data
                $shippingItem->addData($shippingItemsData[$shippingItem->getOrderItemId()])->save();
                unset($shippingItemsData[$shippingItem->getOrderItemId()]);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }

        if(count($shippingItemsData)) {
            foreach($shippingItemsData as $shippingItemToCreateData) {
                try {
                    //if shipping item doesnt exist, create it with order item data
                    $this->shipmentItemFactory->create()->setData([
                        'parent_id' => $shipmentId,
                        'row_total' => $shippingItemToCreateData['row_total'],
                        'price' => $shippingItemToCreateData['price'],
                        'weight' => $shippingItemToCreateData['weight'],
                        'qty' => $shippingItemToCreateData['qty_invoiced'],
                        'product_id' => $shippingItemToCreateData['product_id'],
                        'order_item_id' => $shippingItemToCreateData['item_id'],
                        'additional_data' => $shippingItemToCreateData['additional_data'],
                        'description' => $shippingItemToCreateData['description'],
                        'name' => $shippingItemToCreateData['name'],
                        'sku' => $shippingItemToCreateData['sku'],
                    ])->save();
                } catch (\Exception $e) {
                    $this->logger->critical($e->getMessage());
                }
            }
        }

        try {
            //Update shipment missing information
            if(!$order->getCustomerIsGuest()) {
                $shipment->setCustomerId($order->getCustomerId());
            }
            $shipment->setStoreId($order->getStoreId())
                ->setTotalQty($order->getTotalQtyOrdered())
                ->setShippingAddressId($order->getBillingAddressId())
                ->setBillingAddressId($order->getShippingAddressId())
                ->save();
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return $result;
    }
}
