<?php
/**
 * Author: Simone Monterubbiano <s.monterubbiano@drop.it>
 * Date: 27/02/2019
 * File name: Shipment.php
 * Project: drop/module-gsped
 */
namespace Drop\Gsped\Model\Gsped;

use \Magento\Framework\App\Filesystem\DirectoryList;

class Shipment implements \Drop\Gsped\Api\Data\GspedInterface {

    const GSPED_DIR = "gsped";
    const SINK_DIR = "response_body";
    private $gsped_server_path;


    /**
     * @var Client
     */
    private $client;
    /**
     * @var \Drop\Gsped\Helper\Data
     */
    private $helper;
    /**
     * @var \Drop\Gsped\Logger\Logger
     */
    private $logger;
    /**
     * @var \Magento\Framework\Filesystem
     */
    private $filesystem;

    /**
     * Shipment constructor.
     * @param Client $client
     */
    public function __construct(
        \GuzzleHttp\Client $client,
        \Drop\Gsped\Helper\Data $helper,
        \Drop\Gsped\Logger\Logger $logger,
        \Magento\Framework\Filesystem $filesystem
    )
    {
        $this->client = $client;
        $this->helper = $helper;
        $this->logger = $logger;
        $this->filesystem = $filesystem;

        // Set server path to gsped folder
        $this->gsped_server_path = $this->filesystem->getDirectoryRead(DirectoryList::PUB)->getAbsolutePath().self::GSPED_DIR;
    }


    /**
     * Create shipment on Gsped
     * @param array $data
     * @return $this|array|mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function createShipment(array $data){

        if(!$this->helper->getActive()){
            throw new \Exception("Gsped is disabled from backend");
        }

        if(!is_array($data)){
            throw new \Exception("Data passed is not an array");
        }

        if ($data["peso"] <= 0) {
            $data["peso"] = 0.01;
        }

        // Collect all params
        $params = $this->prepareParams($data);

        try {
            // Make request
            $response = $this->client->request("POST",$this->helper->getApiBaseUrl()."pickup", $params );
        }catch (\Exception $e){
            throw new \Exception($e->getMessage());
        }

        switch($response->getStatusCode()):
            case 200:
                $body = json_decode($response->getBody());
                return $this->processResponse($body);
            case 400:
                throw new \Exception("Malformed request passed");
            case 500:
                throw new \Exception("General error while creating shipment");
        endswitch;
    }

    /**
     * Get shipment from gsped by gsped shipment id
     * @param integer $gspedShipmentId
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getShipment($gspedShipmentId){
        if(empty($gspedShipmentId)){
            throw new \Exception("Gsped shipment id is empty");
        }

        // Collect all params
        $params = $this->prepareParams();

        try {
            // Make request
            $response = $this->client->request("GET", $this->helper->getApiBaseUrl()."pickup?id=".$gspedShipmentId,$params);
        }catch (\Exception $e){
            throw new \Exception($e->getMessage());
        }

        switch($response->getStatusCode()):
            case 200:
                return json_decode($response->getBody());
            case 400:
                throw new \Exception("Gsped shipment id is empty or request is wrong");
            case 404:
                throw new \Exception("Shipment not exists");
            case 500:
                throw new \Exception("General error while getting shipment");
        endswitch;
    }

    /**
     * Get tracking shipment info by ddt_num
     * @param $ddtNum
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getTracking($ddtNum){
        if(empty($ddtNum)){
            throw new \Exception("ddt_num is empty");
        }

        // Collect all params
        $params = $this->prepareParams();

        try {
            // Make request
            $response = $this->client->request("GET", $this->helper->getApiBaseUrl()."tracking?ddt_num=".$ddtNum,$params);
        }catch (\Exception $e){
            throw new \Exception($e->getMessage());
        }

        switch($response->getStatusCode()):
            case 200:
                return json_decode($response->getBody());
            case 400:
                throw new \Exception("Gsped shipment id is empty or request is wrong");
            case 404:
                throw new \Exception("Shipment not exists");
            case 500:
                throw new \Exception("General error while getting shipment");
        endswitch;
    }

    /**
     * TODO
     */
    public function getRate( array $data){
        if(!is_array($data)){
            throw new \Exception("Data passed is not an array");
        }

        // Collect all params
//        $params = $this->prepareParams($data);
//
//        try {
//            // Make request
//            $response = $this->client->request("GET", $this->helper->getApiBaseUrl()."rate",$params);
//        }catch (\Exception $e){
//            throw new \Exception($e->getMessage());
//        }
//
//        switch($response->getStatusCode()):
//            case 200:
//                return json_decode($response->getBody());
//            case 400:
//                throw new \Exception("Gsped shipment id is empty or request is wrong");
//            case 404:
//                throw new \Exception("Shipment not exists");
//            case 500:
//                throw new \Exception("General error while getting shipment");
//        endswitch;
    }

    /**
     * Prepare all params to send to Gsped
     * @param $data
     * @return array
     */
    private function prepareParams(array $data=[]){

//        $data = array_merge($data,["client_id"=>$this->helper->getClientId()]);

        // Header
        $params = [
            "headers" => [
                "Content-Type" => "application/json",
                "Accept"     => "application/json",
                "X-API-KEY"      => $this->helper->getApiKey()
            ],
            "json" => $data,
            "connect_timeout" => $this->helper->getConnectionTimeout(),
            "timeout" => $this->helper->getTimeout(),
            "read_timeout" => $this->helper->getReadTimeout(),

        ];

        // Add synk guzzle param if enabled on backend
        if($this->helper->getEnableSynkLog()){
            $params = array_merge($params,["sink" => $this->gsped_server_path.DIRECTORY_SEPARATOR.self::SINK_DIR.DIRECTORY_SEPARATOR.time().".json"]);
        }

        return $params;
    }


    /**
     * @param $body
     * @return array
     * @throws \Exception
     */
    private function processResponse($body){

        // If enabled save label to filesystem and add label_path field to body object returned
        if($this->helper->getSaveLabel()) {
            $labelsPath = [];
            // Get labels from response body
            $labels = $body->label_pdf;

            if (!count( $labels )) {
                $msgError = "No labels for rem {$body->ddt_alpha}";
                $this->logger->error( "No labels for rem {$body->ddt_alpha}" );
                throw new \Exception( $msgError );
            }

            foreach ($labels as $index => $label):
                $fileName = $this->saveLabel($body, $label, $index);
                $labelsPath[] = $this->gsped_server_path . DIRECTORY_SEPARATOR . $fileName;
            endforeach;
            $body->label_path = $labelsPath;
        }

        return $body;
    }

    /**
     * @param $body
     * @param $label
     * @param $index
     * @return string
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    private function saveLabel( $body, $label, $index){
        // Filename
        $fileName = date("Ymd_His")."_".$body->ddt_num . "_" . $index . ".pdf";

        // Decode label
        $decoded = base64_decode( $label );

        // Set directory
        $writer = $this->filesystem->getDirectoryWrite( DirectoryList::PUB );

        // Create label file and write
        $file = $writer->openFile( self::GSPED_DIR . DIRECTORY_SEPARATOR . $fileName, 'w' );
        try {
            $file->lock();
            try {
                $file->write( $decoded );
            } finally {
                $file->unlock();
            }
        } finally {
            $file->close();
        }

        return $fileName;
    }
}
