<?php
/**
 * Created by PhpStorm.
 * User: Alberto Pallotto
 * Date: 12/06/2019
 * Time: 14:12
 */

namespace Drop\GoogleCloud\Model;

use \Google\Client;
use \Google\Service\Compute;

class Update
{
    const XML_HTTP_CACHE_HOSTS = 'http_cache_hosts';
    const PORT = '81';

    /**
     * @var \Drop\GoogleCloud\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\Framework\App\DeploymentConfig
     */
    private $deploymentConfig;
    /**
     * @var \Magento\Setup\Model\ConfigGenerator
     */
    private $configGenerator;
    /**
     * @var \Magento\Framework\App\DeploymentConfig\Writer
     */
    private $writer;

    /**
     * Update constructor.
     * @param \Drop\GoogleCloud\Helper\Data $helper
     * @param \Magento\Framework\App\DeploymentConfig $deploymentConfig
     * @param \Magento\Setup\Model\ConfigGenerator $configGenerator
     * @param \Magento\Framework\App\DeploymentConfig\Writer $writer
     */
    public function __construct(
        \Drop\GoogleCloud\Helper\Data $helper,
        \Magento\Framework\App\DeploymentConfig $deploymentConfig,
        \Magento\Setup\Model\ConfigGenerator $configGenerator,
        \Magento\Framework\App\DeploymentConfig\Writer $writer
    )
    {
        $this->helper = $helper;
        $this->deploymentConfig = $deploymentConfig;
        $this->configGenerator = $configGenerator;
        $this->writer = $writer;
    }

    /**
     * @return $this|void
     */
    public function process()
    {
        $gcServer = $this->getGoogleCloudData();
        $httpCacheHosts = $this->getDeploymentConfiguration();

        sort($gcServer['ips']);
        sort($httpCacheHosts['ips']);

        if ($gcServer['ips'] == $httpCacheHosts['ips']) {
            return;
        }

        $newHttpCacheHosts = [];
        foreach ($gcServer['instances'] as $instance) {
            $newHttpCacheHosts[] = [
                'host' => $instance['ip'],
                'port' => self::PORT
            ];
        }

        $config = [
            \Magento\Framework\Config\File\ConfigFilePool::APP_ENV => [
                \Magento\Framework\Config\ConfigOptionsListConstants::CONFIG_PATH_CACHE_HOSTS => $newHttpCacheHosts
            ]
        ];
        try{
            $this->writer->saveConfig($config, true);
        }catch(\Exception $e){
            throw new \Exception($e->getMessage());
        }

        return $this;
    }

    /**
     * @return array
     */
    public function getDeploymentConfiguration()
    {
        $returns = [];
        $hosts = $this->deploymentConfig->get(self::XML_HTTP_CACHE_HOSTS);

        foreach ($hosts as $host):
            $returns['ips'][] = $host['host'];
        endforeach;
        $returns['instances'] = $hosts;

        return $returns;
    }

    public function getGoogleCloudData()
    {
        $client = new \Google_Client();

        // application default credentials
        $client->useApplicationDefaultCredentials();
        // set the information from the config
        $client->setConfig('type', $this->helper->getAccountType());
        $client->setConfig('signing_key', $this->helper->getPrivateKey());
        $client->setConfig('client_email', $this->helper->getClientEmail());
        $client->setConfig('signing_algorithm', 'HS256');

        $client->addScope('https://www.googleapis.com/auth/cloud-platform');

        $service = new \Google_Service_Compute($client);
        $elaboratedResponse = [];
        do {
            $response = $service->instances->listInstances($this->helper->getProjectId(), $this->helper->getZone(), []);

            foreach ($response['items'] as $instance) {
                if (strpos($instance->getName(), 'autoscaling-cluster1-') === false) {
                    continue;
                }

                $modelData = $this->accessProtected($instance, 'modelData');
                if (!isset($modelData['networkInterfaces'][0]['networkIP'])) {
                    continue;
                }

                $elaboratedResponse['ips'][] = $modelData['networkInterfaces'][0]['networkIP'];
                $elaboratedResponse['instances'][] = [
                    'name' => $instance->getName(),
                    'ip'   => $modelData['networkInterfaces'][0]['networkIP']
                ];
            }

            $optParams['pageToken'] = $response->getNextPageToken();
        } while ($optParams['pageToken']);

        return $elaboratedResponse;
    }


    public function accessProtected($obj, $prop)
    {
        $reflection = new \ReflectionClass($obj);
        $property = $reflection->getProperty($prop);
        $property->setAccessible(true);
        return $property->getValue($obj);
    }
}