<?php
/**
 * Spreadsheets
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\GoogleApiAdapter\Model\Service;

use \Drop\GoogleApiAdapter\Model\AbstractConnection;

class Spreadsheets extends AbstractConnection implements ServiceInterface
{

    public $scope = \Google_Service_Sheets::SPREADSHEETS_READONLY;

    /**
     * @return $this|void
     * @throws \Exception
     */
    public function setService() {
        $this->service = new \Google_Service_Sheets($this->getClient());
        return $this;
    }

    /**
     * https://developers.google.com/sheets/api/referencee/rest/v4/spreadshets/get
     *
     * @param $fileId
     * @return mixed
     */
    public function getSpreadsheet($fileId) {
        if(empty($fileId)) {
            throw new \Exception('Please enter a file ID. The easiest way to get a Google Drive file ID is from Google Drive on the web. Right-click the file name and select "Get shareable link". The last part of the link is the file ID.');
        }
        return $this->getService()->spreadsheets->get($fileId);
    }

    /**
     * https://developers.google.com/sheets/api/reference/rest/v4/spreadsheets/get
     *
     * @param $fileId
     * @return mixed
     */
    public function getSpreadsheetData($fileId) {
        if(empty($fileId)) {
            throw new \Exception('Please enter a file ID. The easiest way to get a Google Drive file ID is from Google Drive on the web. Right-click the file name and select "Get shareable link". The last part of the link is the file ID.');
        }
        return $this->getSpreadsheet()->getValues($fileId);
    }

    /**
     * https://developers.google.com/sheets/api/reference/rest/v4/spreadsheets/get
     *
     * @param $fileId
     * @param $sheetName
     * @return mixed
     * @throws \Exception
     */
    public function getSingleSheetSpreadsheet($fileId, $sheetName) {
        if(empty($fileId)) {
            throw new \Exception('Please enter a file ID. The easiest way to get a Google Drive file ID is from Google Drive on the web. Right-click the file name and select "Get shareable link". The last part of the link is the file ID.');
        }
        if(empty($sheetName)) {
            throw new \Exception('Please enter a Range in A1 notation (https://developers.google.com/sheets/api/guides/concepts)');
        }
        return $this->getService()->spreadsheets_values->get($fileId, $sheetName);
    }

    /**
     * https://developers.google.com/sheets/api/reference/rest/v4/spreadsheets/get
     *
     * @param $fileId
     * @param $sheetName
     * @return mixed
     * @throws \Exception
     */
    public function getSingleSheetSpreadsheetValues($fileId, $sheetName) {
        if(empty($fileId)) {
            throw new \Exception('Please enter a file ID. The easiest way to get a Google Drive file ID is from Google Drive on the web. Right-click the file name and select "Get shareable link". The last part of the link is the file ID.');
        }
        if(empty($sheetName)) {
            throw new \Exception('Please enter a Range in A1 notation (https://developers.google.com/sheets/api/guides/concepts)');
        }
        return $this->getSingleSheetSpreadsheet($fileId, $sheetName)->getValues();
    }

}
