<?php
/**
 * Drive
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\GoogleApiAdapter\Model\Service;

use \Drop\GoogleApiAdapter\Model\AbstractConnection;

class Drive extends AbstractConnection implements ServiceInterface
{

    public $scope = \Google_Service_Sheets::DRIVE_READONLY;

    /**
     * @return $this|void
     * @throws \Exception
     */
    public function setService() {
        $this->service = new \Google_Service_Drive($this->getClient());
        return $this;
    }

    /**
     * https://developers.google.com/drive/api/v2/reference/files/list
     *
     * @return mixed
     * @throws \Exception
     */
    public function getFilesList() {
        $files = $this->getService()->files->listFiles(array());
        return $files->getFiles();
    }

    /**
     * https://developers.google.com/drive/api/v3/manage-downloads
     *
     * @param $fileId
     * @return mixed
     * @throws \Exception
     */
    public function getFile($fileId) {
        if(empty($fileId)) {
            throw new \Exception('Please enter a file ID. The easiest way to get a Google Drive file ID is from Google Drive on the web. Right-click the file name and select "Get shareable link". The last part of the link is the file ID.');
        }
        return $this->getService()->files->get($fileId, ['alt' => 'media']);
    }

    /**
     * @param $fileId
     * @return bool
     * @throws \Exception
     */
    public function downloadFile($fileId) {
        $file = $this->getFile();

        $downloadUrl = $file->getDownloadUrl();
        if ($downloadUrl) {
            $request = new \Google_Http_Request($downloadUrl, 'GET', null, null);
            $httpRequest = $this->getService()->getAuth()->authenticatedRequest($request);
            if ($httpRequest->getResponseHttpCode() == 200) {
                return $httpRequest->getResponseBody();
            }
        }
        return false;
    }

}
