<?php
/**
 * AbstractConnection
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\GoogleApiAdapter\Model;

abstract class AbstractConnection
{

    /**
     * @var \Drop\GoogleApiAdapter\Helper\Data
     */
    private $helper;
    /**
     * @var \Google_Client
     */
    private $client;

    public $scope;
    public $service;

    /**
     * AbstractConnection constructor.
     *
     * @param \Drop\GoogleApiAdapter\Helper\Data $helper
     * @param $service
     */
    public function __construct(
        \Drop\GoogleApiAdapter\Helper\Data $helper
    )
    {
        $this->helper = $helper;
    }

    /**
     * @return array
     * @throws \Exception
     */
    public function authorize()
    {
        $this->initClient();

        $accessToken = $this->helper->getAccessToken();
        if(empty($accessToken)) {
            return $this->client->createAuthUrl();
        }

        $this->setAccessToken();
        return true;
    }

    /**
     * @return bool|string
     * @throws \Exception
     */
    public function revoke()
    {
        $this->initClient();

        $accessToken = $this->helper->getAccessToken();
        if(!empty($accessToken)) {
            $this->client->revokeToken($accessToken);
            $this->helper->setAccessToken(null);
        }
        return true;
    }

    /**
     * @return \Google_Client
     * @throws \Exception
     */
    public function getClient()
    {
        $this->initClient()
            ->setAccessToken();
        return $this->client;
    }

    /**
     * @param $authCode
     * @return $this
     * @throws \Exception
     */
    public function setAccessTokenByAuthCode($authCode)
    {
        $accessToken = $this->client->fetchAccessTokenWithAuthCode($authCode);
        if (array_key_exists('error', $accessToken)) {
            throw new \Exception(join(', ', $accessToken));
        }

        $this->helper->setAccessToken($accessToken);
        return $this;
    }

    /**
     * @return $this
     * @throws \Exception
     */
    public function initClient()
    {
        $this->client = new \Google_Client();

        if(empty($this->scope)) {
            $this->scope = $this->helper->getAllScopesAsArray();
        }
        $this->client->setScopes($this->scope);

        $credentialJson = $this->helper->getCredentialJson();
        if(empty($credentialJson) || empty($credentialJson['web'])) {
            throw new \Exception('Not uploaded credential Json. Please follow the instruction in configuration page');
        }

        $this->client->setClientId($credentialJson['web']['client_id']);
        if (isset($credentialJson['web']['client_secret'])) {
            $this->client->setClientSecret($credentialJson['web']['client_secret']);
        }
        if (isset($credentialJson['web']['redirect_uris'])) {
            $this->client->setRedirectUri($credentialJson['web']['redirect_uris'][0]);
        } else {
            throw new \Exception('Not set Redirect uri on console cloud. Please follow the instruction in configuration page');
        }

        $this->client->setAccessType('offline');
        $this->client->setApprovalPrompt('force');

        return $this;
    }

    /**
     * @return $this
     */
    public function setAccessToken()
    {
        $accessToken = $this->helper->getAccessToken();
        $this->client->setAccessToken($accessToken);

        //Check if renew is needed
        if ($this->client->isAccessTokenExpired()) {
            $this->client->fetchAccessTokenWithRefreshToken($this->client->getRefreshToken());
            $this->helper->setAccessToken($this->client->getAccessToken());
        }

        return $this;
    }

    /**
     * @return mixed
     * @throws \Exception
     */
    public function getService() {
        if(empty($this->service)) {
            $this->setService();
        }
        return $this->service;
    }

    public function setService() {
        throw new \Exception('Please make sure you are extending a Service.');
    }
}
