<?php
/**
 * Data
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\GoogleApiAdapter\Helper;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{

    const XML_PATH_GOOGLE_API_CREDENTIAL_JSON_UPLOADER = 'google_api_adapter/configuration/credential_json_uploader';
    const XML_PATH_GOOGLE_API_CREDENTIAL_JSON = 'google_api_adapter/configuration/credential_json';
    const XML_PATH_GOOGLE_API_ACCESS_TOKEN = 'google_api_adapter/configuration/access_token';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var \Magento\Framework\App\Config\Storage\WriterInterface
     */
    private $configWriter;
    /**
     * @var \Magento\Framework\Encryption\EncryptorInterface
     */
    private $encryptor;
    /**
     * @var \Magento\Framework\App\Cache\TypeListInterface
     */
    private $cacheTypeList;

    /**
     * Data constructor.
     *
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Config\Storage\WriterInterface $configWriter
     * @param \Magento\Framework\App\Cache\TypeListInterface $cacheTypeList
     * @param \Magento\Framework\Encryption\EncryptorInterface $encryptor
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Config\Storage\WriterInterface $configWriter,
        \Magento\Framework\App\Cache\TypeListInterface $cacheTypeList,
        \Magento\Framework\Encryption\EncryptorInterface $encryptor
    )
    {
        $this->scopeConfig = $context->getScopeConfig();
        $this->configWriter = $configWriter;
        $this->cacheTypeList = $cacheTypeList;
        $this->encryptor = $encryptor;
    }

    /**
     * @return mixed
     */
    public function getCredentialJsonUploader() {
        return $this->scopeConfig->getValue(self::XML_PATH_GOOGLE_API_CREDENTIAL_JSON_UPLOADER);
    }

    /**
     * @param $accessToken
     */
    public function setCredentialJson($credentialJson) {
        if(!empty($credentialJson)) {
            $credentialJson = $this->encryptor->encrypt($credentialJson);
        }
        $this->configWriter->save(self::XML_PATH_GOOGLE_API_CREDENTIAL_JSON, $credentialJson);
        $this->cacheTypeList->cleanType(\Magento\Framework\App\Cache\Type\Config::TYPE_IDENTIFIER);
        $this->cacheTypeList->cleanType(\Magento\PageCache\Model\Cache\Type::TYPE_IDENTIFIER);
        return $this;
    }

    /**
     * @return mixed
     */
    public function getCredentialJson() {
        return json_decode($this->getCredentialJsonAsString(), true);
    }

    /**
     * @return string
     */
    public function getCredentialJsonAsString() {
        return $this->encryptor->decrypt($this->scopeConfig->getValue(self::XML_PATH_GOOGLE_API_CREDENTIAL_JSON));
    }

    /**
     * @param $accessToken
     * @return $this
     */
    public function setAccessToken($accessToken) {
        if(!empty($accessToken)) {
            $accessToken = $this->encryptor->encrypt(json_encode($accessToken));
        }
        $this->configWriter->save(self::XML_PATH_GOOGLE_API_ACCESS_TOKEN, $accessToken);
        $this->cacheTypeList->cleanType(\Magento\Framework\App\Cache\Type\Config::TYPE_IDENTIFIER);
        $this->cacheTypeList->cleanType(\Magento\PageCache\Model\Cache\Type::TYPE_IDENTIFIER);
        return $this;
    }

    /**
     * @return mixed
     */
    public function getAccessToken() {
        return json_decode($this->getAccessTokenAsString(), true);
    }

    /**
     * @return string
     */
    public function getAccessTokenAsString() {
        return $this->encryptor->decrypt($this->scopeConfig->getValue(self::XML_PATH_GOOGLE_API_ACCESS_TOKEN));
    }

    /**
     * @return bool
     */
    public function getGoogleApiIsConnected()
    {
        if($this->getCredentialJsonUploader() && $this->getAccessToken()) {
            return true;
        }
        return false;
    }

    /**
     * @return string
     */
    public function getDriveScope() {
        return \Google_Service_Sheets::DRIVE_READONLY;
    }

    /**
     * @return string
     */
    public function getSpreadsheetsScope() {
        return \Google_Service_Sheets::SPREADSHEETS_READONLY;
    }

    /**
     * @return array
     */
    public function getAllScopesAsArray() {
        return [
            $this->getDriveScope(),
            $this->getSpreadsheetsScope()
        ];
    }


}
