<?php

/*
 * Description: OrderPlaceAfter.php
 * @date: 16-mag-2018
 * @author Simone Monterubbiano <s.monterubbiano@drop.it>
 */

namespace Drop\Gestionale\Model\Consumer\Order;

use Drop\Gestionale\Model\XmlGenerator;

class PlaceAfter implements \Rcason\Mq\Api\ConsumerInterface  {
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;
    /**
     * @var XmlGenerator
     */
    private $xmlGenerator;
    /**
     * @var \Magento\Sales\Model\OrderRepository
     */
    private $orderRepository;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $file;
    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $dir;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;

    /**
     * PlaceAfter constructor.
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Helper\Data $helper
     * @param XmlGenerator $xmlGenerator
     * @param \Magento\Sales\Model\OrderRepository $orderRepository
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     */
    public function __construct(
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Helper\Data $helper,
        \Drop\Gestionale\Model\XmlGenerator $xmlGenerator,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Framework\Filesystem\Io\File $file,
        \Magento\Framework\Filesystem\DirectoryList $dir,
        \Magento\Framework\Stdlib\DateTime\DateTime $date
    ) {

        $this->logger = $logger;
        $this->helper = $helper;
        $this->xmlGenerator = $xmlGenerator;
        $this->orderRepository = $orderRepository;
        $this->productRepository = $productRepository;
        $this->file = $file;
        $this->dir = $dir;
        $this->date = $date;
    }

    /**
     * If order has an offline payment method, export
     * @param $orderId
     * @return void
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function process($orderId) {
        if(empty($orderId)){
            $this->logger->error("[GENERATE-XML-FILE] - orderId is empty, return",["method"=>__METHOD__]);
            return;
        }

        // Load order by increment id
        $order = $this->orderRepository->get($orderId);

        // If order is not an object, return
        if (!is_object($order)) {
            $this->logger->error("[GENERATE-XML-FILE] - Order is not an object, return",["method"=>__METHOD__]);
            return;
        }

        // Check if order contains products to send to GESTIONALE (ship_tws = 1)
        $toSend=0;
        foreach($order->getAllVisibleItems() as $item):
            if(isset($item["product_id"])){
                $simpleId = $item["product_id"];
            }elseif (isset($item["product_options"]["info_buyRequest"]["selected_configurable_option"])) {
                $simpleId = $item["product_options"]["info_buyRequest"]["selected_configurable_option"];
            }

            if(!isset($simpleId)){continue;}

            $itemObj = $this->productRepository->getById($simpleId);
            if($itemObj->getShipGestionale()){$toSend++;}
        endforeach;

        if($toSend==0){
            $this->logger->error("[GENERATE-XML-FILE] - Order contains only personalized products, return ",["method"=>__METHOD__]);
            return;
        }

        // Check payment code
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentCode = (string)$method->getCode();

        $incrementId = str_replace("-","",$order->getIncrementId());

        // Full path file name
        $dateTime = $this->date->date("YmdHis");
        $fileName = "OUT_940_ITA3PL_".$dateTime."_".$incrementId.".ITA";
        $localFilePath = $this->dir->getRoot() . $this->helper->getOrdersLocalPath() . $fileName;

        // If offline payment, generate XML
        if (in_array($paymentCode, $this->helper->getOfflineMethods())) {
            $xml = $this->xmlGenerator->generateOrderXml($order);
        }else{
            return;
        }

        // Save order XML to file
        try {
            $this->file->write( $localFilePath, $xml, 0666 );
            $this->logger->info("[GENERATE-XML-FILE] - Order created",["incrementId"=>$order->getIncrementId(),"method"=>__METHOD__]);
        }catch (\Exception $e){
            $this->logger->critical("[GENERATE-XML-FILE] - Error creating file",["exception"=>$e->getMessage(),"incrementId"=>$order->getIncrementId(), "method"=>__METHOD__]);
        }

        return;
    }
}