<?php

/*
 * Description: OrderPlaceAfter.php
 * @date: 16-mag-2018
 * @author Simone Monterubbiano <s.monterubbiano@drop.it>
 */

namespace Drop\Gestionale\Model\Consumer\Order;

class InvoiceRegister implements \Rcason\Mq\Api\ConsumerInterface {
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\Sales\Model\OrderRepository
     */
    private $orderRepository;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $file;
    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $dir;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;
    /**
     * @var \Drop\Gestionale\Model\XmlGenerator
     */
    private $xmlGenerator;

    /**
     * InvoiceRegister constructor.
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Helper\Data $helper
     * @param \Drop\Gestionale\Model\XmlGenerator $xmlGenerator
     * @param \Magento\Sales\Model\OrderRepository $orderRepository
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     */
    public function __construct(
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Helper\Data $helper,
        \Drop\Gestionale\Model\XmlGenerator $xmlGenerator,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Framework\Filesystem\Io\File $file,
        \Magento\Framework\Filesystem\DirectoryList $dir,
        \Magento\Framework\Stdlib\DateTime\DateTime $date
    ) {

        $this->logger = $logger;
        $this->helper = $helper;
        $this->orderRepository = $orderRepository;
        $this->productRepository = $productRepository;
        $this->file = $file;
        $this->dir = $dir;
        $this->date = $date;
        $this->xmlGenerator = $xmlGenerator;
    }

    public function process($orderId) {

        // If incrementId is empty, return
        if(empty($orderId)){
            $this->logger->error("[GENERATE-XML-FILE] - Order incrementId is empty",["method"=>__METHOD__]);
            return $this;
        }

        // Load order by increment id
        $order = $this->orderRepository->get($orderId);

        // Check if is object
        if(!is_object($order)){
            $this->logger->error("[GENERATE-XML-FILE] - Order #{$order->getIncrementId()} is not an object, return",["method"=>__METHOD__]);
            return $this;
        }

        // Check if order contains products to send to GESTIONALE (ship_tws = 1)
        $toSend=0;
        foreach($order->getAllVisibleItems() as $item):
            if(isset($item["product_id"])){
                $simpleId = $item["product_id"];
            }elseif (isset($item["product_options"]["info_buyRequest"]["selected_configurable_option"])) {
                $simpleId = $item["product_options"]["info_buyRequest"]["selected_configurable_option"];
            }

            if(!isset($simpleId)){continue;}

            $itemObj = $this->productRepository->getById($simpleId);
            if($itemObj->getShipGestionale()){$toSend++;}
        endforeach;

        if($toSend==0){
            $this->logger->error("[GENERATE-XML-FILE] - Order contains only personalized products, return ",["method"=>__METHOD__]);
            return;
        }
        
        // Check if order has invoice
        if(!$order->hasInvoices()){
            $this->logger->error("[GENERATE-XML-FILE] - Order has not invoice, return",["method"=>__METHOD__]);
            return $this;
        }

        // Check payment code, if offline payment, export
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentCode = $method->getCode();

        // If an offline method, export 941 file to confirm order, else is a online payment so export 940 file
        if (in_array($paymentCode, $this->helper->getOfflineMethods())) {
            $type = "941";
            $xml = $this->xmlGenerator->generateConfirmOrderXml($order); // 941 file (offline payment)
        }else{
            $type = "940";
            $xml = $this->xmlGenerator->generateOrderXml($order); // 940 file (online payment)
        }

        $incrementId = str_replace("-","",$order->getIncrementId());

        // File name where saving XML content
        $dateTime = $this->date->date("YmdHis");
        $fileName = "OUT_{$type}_ITA3PL_".$dateTime."_".$incrementId.".ITA";
        $localFilePath = $this->dir->getRoot() . $this->helper->getOrdersLocalPath() . $fileName;

        // Save order XML to file
        try {
            $this->file->write( $localFilePath, $xml, 0666 );
            $this->logger->info("[GENERATE-XML-FILE] - Invoice created",[
                    "incrementId"=>$order->getIncrementId(),
                    "method"=>__METHOD__]
            );
        }catch (\Exception $e){
            $this->logger->critical("[GENERATE-XML-FILE] - ERROR generating XML file",[
                    "incrementId"=>$order->getIncrementId(),
                    "method"=>__METHOD__,
                    "exception" => $e->getMessage()]
            );
        }

        return;
    }
}