<?php

/*
 * Description: Gestionale.php
 * @date: 14-mag-2018
 * @author Simone Monterubbiano <s.monterubbiano@drop.it>
 */

namespace Drop\Gestionale\Model\Consumer\Ftp;


class SendXml implements \Rcason\Mq\Api\ConsumerInterface {

    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $dir;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Gestionale\Model\GestionaleFtp
     */
    private $twsFtp;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;


    /**
     * SendXml constructor.
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Model\GestionaleFtp $twsFtp
     * @param \Drop\Gestionale\Helper\Data $helper
     */
    public function __construct(
        // Magento
        \Magento\Framework\Filesystem\DirectoryList $dir, 
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\Message\ManagerInterface $messageManager,

        // Drop
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Model\GestionaleFtp $twsFtp,
        \Drop\Gestionale\Helper\Data $helper
    ) {

        $this->dir = $dir;
        $this->date = $date;
        $this->logger = $logger;
        $this->twsFtp = $twsFtp;
        $this->messageManager = $messageManager;
        $this->helper = $helper;
    }

    public function process($message) {
        // Check if GESTIONALE enabled
        if(!$this->helper->isGestionaleEnabled()){
            return false;
        }

        // If there are not files to send, return false
        if(!count($this->getFilesToSend())){
            $this->logger->info("[SEND-XML] - There are no files to send",["method"=>__METHOD__]);
            $this->messageManager->addWarningMessage("There are no files to send, exit");
            return false;
        }

        // Check semaphore file, if exists, return false
        if($this->twsFtp->semaphoreExists($this->helper->getOrdersRemoteImportPath())){
            $this->logger->info("[SEND-XML] - Semaphore, can't upload",["method"=>__METHOD__]);
            $this->messageManager->addWarningMessage("There is a semaphore file, not upload files, exit");
            return false;
        }

        // Start to send files
        foreach ($this->getFilesToSend() as $file):

            // Rename file: obsolete
            //$newFileName = $this->renameFile($file);
            $explode = explode("/",$file);

            try {
                // Send via FTP
                $this->twsFtp->upload(end($explode), $file, $this->helper->getOrdersRemoteImportPath());

                // Move file to processed
                rename($file, $this->dir->getRoot() . $this->helper->getOrdersLocalProcessedPath() . end($explode));

                // Log action
                $this->messageManager->addSuccessMessage("Uploaded file {$file} to FTP");
            } catch (\Exception $e) {
                $this->messageManager->addComplexErrorMessage("Error while sending files",["exception"=>$e->getMessage()]);
                $this->logger->error("[SEND-XML] - ".$e->getMessage(),["method"=>__METHOD__]);
            }
        endforeach;

        // Files sent, write semaphore file
        $this->twsFtp->writeSemaphore($this->dir->getRoot() . DIRECTORY_SEPARATOR. $this->helper->getOrdersLocalPath().DIRECTORY_SEPARATOR."TransferOk",$this->helper->getOrdersRemoteImportPath());
        $this->messageManager->addSuccessMessage("Semaphore file created on remote server");
    }

    /**
     * Get all XML files from directory
     * @return array
     */
    public function getFilesToSend() {
        $absolutePath = $this->dir->getRoot() . $this->helper->getOrdersLocalPath();
        return glob("$absolutePath*.{ITA}", GLOB_BRACE);
    }

    /**
     * Rename file with correct extension and syntax
     * @param string $filename
     * @return string New file name
     */
    private function renameFile($filename) {
        $newFileName = substr_replace($filename, 'ITA', strrpos($filename, '.') + 1);
        $newFileName = str_replace("-", "_", $newFileName);
        rename($filename, $newFileName);
        return $newFileName;
    }
}
