<?php
/**
 * Author: Simone Monterubbiano <s.monterubbiano@drop.it>
 * Date: 04/09/2018
 * File name: XmlGenerator.php
 * Project: JustEat
 */
namespace Drop\Gestionale\Model;

class XmlGenerator {
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\Sales\Model\OrderRepository
     */
    private $orderRepository;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * XmlGenerator constructor.
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Helper\Data $helper
     * @param \Magento\Sales\Model\OrderRepository $orderRepository
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     */
    public function __construct(
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Helper\Data $helper,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
    )
    {
        $this->logger = $logger;
        $this->helper = $helper;
        $this->orderRepository = $orderRepository;
        $this->productRepository = $productRepository;
    }

    /**
     * GENERATE 940 files
     * @param $order
     * @return mixed
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function generateOrderXml($order){
        $xml = new \SimpleXMLElement('<Envelope></Envelope>');

        // Order date
        $createdAt = date_format(date_create($order->getCreatedAt()), 'Ymd');
//        echo "<pre>";
//        print_r($billingAddress = $order->getShippingAddress()->getData());
//        echo "</pre>";

        // Second and Third Ship Address Row
        if(!empty($order->getShippingAddress()->getStreet()[1])){
            $ShiptoStreet2 = $order->getShippingAddress()->getStreet()[1];
        }

        if(!empty($order->getShippingAddress()->getStreet()[2])){
            $ShiptoStreet2 .= " ".$order->getShippingAddress()->getStreet()[2];
        }

        if(empty($ShiptoStreet2)){
            $ShiptoStreet2 = "";
        }

        // HEADER
        $header = $xml->addChild('Header');
        $header->addChild('TransactionSetIdentifierCode', "940");
        $header->addChild('TransactionSetControlNumber',$order->getId());
        $header->addChild('BlockCode',$this->getBlockCode($order));
        $header->addChild('Location',"GESTIONALEJE");
        $header->addChild('Season',"ALL");
        $header->addChild('Division',"ALL");
        $header->addChild('OrderNum',$order->getIncrementId());
        $header->addChild('PickDate',$createdAt);
        $header->addChild('StartDate',$createdAt);
        $header->addChild('EndDate',$createdAt);
        $header->addChild('PONum','');
        $header->addChild('OrderType','');
        $header->addChild('Contrassegno',$this->getCheckMoneyOrderImport($order));
        $header->addChild('Source','');
        $header->addChild('ShipTerms',$this->getShipTerms($order));
        $header->addChild('CustomerName',$order->getCustomerFirstName() . " ". $order->getCustomerLastName());
        $header->addChild('CustDBA','');

        // Billing information: TODO: CustStreet2, ShiptoStreet2
        $header->addChild('CustStreet1',$billingAddress = $order->getBillingAddress()->getStreet()[0]);
        $header->addChild('CustCity',$billingAddress = $order->getBillingAddress()->getCity());
        $header->addChild('CustState',$billingAddress = $order->getBillingAddress()->getRegionCode());
        $header->addChild('CustZip',$billingAddress = $order->getBillingAddress()->getPostCode());
        $header->addChild('CustCountry',$billingAddress = $order->getBillingAddress()->getCountryId());
        $header->addChild('CustARNum','');
        $header->addChild('CustVATNum',$order->getCustomerTaxvat());

        // Shipping information
        $header->addChild('ShipToLocId','');
        $header->addChild('ShipToName',$order->getShippingAddress()->getFirstname()." ".$order->getShippingAddress()->getLastname());
        $header->addChild('ShiptoStreet1',$order->getShippingAddress()->getStreet()[0]);

        // Second and third ship address row
        $header->addChild('ShiptoStreet2',$ShiptoStreet2);

        $header->addChild('ShiptoCity',$order->getShippingAddress()->getCity());
        $header->addChild('ShiptoState',$order->getShippingAddress()->getRegionCode());
        $header->addChild('ShiptoZip',$order->getShippingAddress()->getPostCode());
        $header->addChild('ShiptoCountry',$order->getShippingAddress()->getCountryId());
        $header->addChild('ShipInstructions','');
        $header->addChild('PhoneNumber',$order->getShippingAddress()->getTelephone());

        // Store information: TODO StoreStreet2
        $header->addChild('StoreName',$order->getShippingAddress()->getFirstname()." ".$order->getShippingAddress()->getLastname());
        $header->addChild('StoreStreet1',$order->getShippingAddress()->getStreet()[0]);

        // Second and third ship address row
        $header->addChild('StoreStreet2',$ShiptoStreet2);

        $header->addChild('StoreCity',$order->getShippingAddress()->getCity());
        $header->addChild('StoreState',$order->getShippingAddress()->getRegionCode());
        $header->addChild('StoreZip',$order->getShippingAddress()->getPostCode());
        $header->addChild('StoreCountry',$order->getShippingAddress()->getCountryId());

        // Party identification
        $partyIdentification = $header->addChild('PartyIdentification');
        $partyIdentification->addChild('PartyIdentifierCode','JE');
        $partyIdentification->addChild('Name','JustEat');

        $body = $xml->addChild('Body');

//        foreach($order->getItems() as $item):
//            echo $item->getSku()."<br/>";
//        endforeach;
//        die(__METHOD__);

        foreach($order->getItems() as $item):

            // Skip bundle producct, only associated products
            if( ($item->getProductType()=="bundle") or ($item->getProductType()=="configurable")){
                continue;
            }

            // Load full product info (for barcode...)
            $itemObj = $this->productRepository->getById($item->getProductId());

            $productObj = $this->getProductByItem($item);
            if(!$productObj){
                return false;
            }

            $detailLine = $body->addChild('DetailLine');
            $detailLine->addChild('AssignedIdentification',$item->getId());
            $detailLine->addChild('Location','GESTIONALEJE');
            $detailLine->addChild('StartDate',$createdAt);
            $detailLine->addChild('EndDate',$createdAt);
            $detailLine->addChild('Season','');
            $detailLine->addChild('Division','');
            $detailLine->addChild('LineSeq',0);
            $detailLine->addChild('LineStatus','P');
            $detailLine->addChild('Style',$productObj->getSkuJusteat());
            $detailLine->addChild('ColorCode','999');
            $detailLine->addChild('LabelCode','');
            $detailLine->addChild('Dimension','');
            $detailLine->addChild('SizeDesc','OS');
            $detailLine->addChild('UPC',$itemObj->getBarcode());
            $detailLine->addChild('Quantity',(int)$item->getQtyOrdered());
        endforeach;

        $dom = new \DomDocument();
        $dom->loadXML($xml->asXML());
        $dom->formatOutput = true;
        $formatedXML = $dom->saveXML();

        // Debug
//        header ("Content-Type:text/xml");
//        echo $formatedXML;
//        die();

        // Return
        return $formatedXML;
        //return $xml->asXML();
    }

    public function generateConfirmOrderXml($order){

        $xml = new \SimpleXMLElement('<Envelope></Envelope>');

        // Header
        $header = $xml->addChild('Header');
        $header->addChild('TransactionSetIdentifierCode','941');

        // PartyIdentification
        $partyIdentification = $header->addChild('PartyIdentification');
        $partyIdentification->addChild('PartyIdentifierCode','JE');
        $partyIdentification->addChild('Name','JustEat');

        // Body
        $body = $xml->addChild('Body');
        $detailLine = $body->addChild('DetailLine');
        $detailLine->addChild('TicketNumber',$order->getId());
        $detailLine->addChild('BlockCode',$this->getBlockCode($order));

        $dom = new \DomDocument();
        $dom->loadXML($xml->asXML());
        $dom->formatOutput = true;
        $formatedXML = $dom->saveXML();

//        Debug
//        header ("Content-Type:text/xml");
//        die();

        // Return XML
        return $formatedXML;
//        echo  $xml->asXML();
    }

    private function getBlockCode($order){
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentCode = $method->getCode();

        // The logic to get block code:
        // If offline method and has invoice => S (sbloccato)
        // If offline method and not has invoice => B (bloccato)
        // If not offline method => S (sbloccato)
        // If order canceled => C (cancellato)
        if(in_array($paymentCode,$this->helper->getOfflineMethods())){
            if($order->hasInvoices()){
                $code = "S";
            }else{
                $code = "B";
            }
        }else{
            $code = "S";
        }

        // Order canceled
        if($order->getStatus()=="canceled"){
            $code = "C";
        }

        return $code;
    }

    private function getShipTerms($order){
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentCode = $method->getCode();
        if(in_array($paymentCode,$this->helper->getCachet())){
            return 4; // FRANCO+C/ASSEGNO
        }else{
            return 1; // Franco
        }
    }

    private function getCheckMoneyOrderImport($order){
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentCode = $method->getCode();
        if(in_array($paymentCode,$this->helper->getCachet())){
            return $order->getGrandTotal();
        }else{
            return "";
        }
    }

    /**
     * @param $item
     * @return bool|\Magento\Catalog\Api\Data\ProductInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getProductByItem($item){
        if(isset($item["product_id"])){
            $simpleId = $item["product_id"];
        }elseif (isset($item["product_options"]["info_buyRequest"]["selected_configurable_option"])) {
            $simpleId = $item["product_options"]["info_buyRequest"]["selected_configurable_option"];
        }

        if(!isset($simpleId)){return false;}

        $itemObj = $this->productRepository->getById($simpleId);
        return $itemObj;
    }
}