<?php
/**
 * Created by Simone Monterubbiano.
 * Email: s.monterubbiano@drop.it
 * Date: 13/07/2018
 * File name: GestionaleFtp.php
 * Project: JustEat
 */
namespace Drop\Gestionale\Model;

class GestionaleFtp {

    // Constants
    const PROFILE_ID = 5;
    const FTP_PASSIVE = 1;
    const FTP_IMPORT_PATH = "IMPORT";
    const SEMAPHORE_FILE = "TransferOk";
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Magento\Framework\Filesystem\Io\Ftp
     */
    private $mageFtp;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;


    /**
     * GestionaleFtp constructor.
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Magento\Framework\Filesystem\Io\Ftp $mageFtp
     * @param \Drop\Gestionale\Helper\Data $helper
     */
    public function __construct(
        \Drop\Gestionale\Logger\Logger $logger,
        \Magento\Framework\Filesystem\Io\Ftp $mageFtp,
        \Drop\Gestionale\Helper\Data $helper
    )
    {
        $this->logger = $logger;
        $this->mageFtp = $mageFtp;
        $this->helper = $helper;
    }

    /**
     * @param $ftpPath
     * @return bool|true
     */
    private function ftpOpen($ftpPath){
        try {
            return $this->mageFtp->open(
                [
                    'host' => $this->helper->getFtpHost(),
                    'user' => $this->helper->getFtpUsername(),
                    'username' => $this->helper->getFtpUsername(),
                    'password' => $this->helper->getFtpPassword(),
                    'path' => $ftpPath,
                    'passive' => $this->helper->getFtpUsePassiveMove(),
                    'port' => null,
                    'timeout' => '120',
                ]
            );
        }catch (\Exception $e){
            $this->logger->critical("Cannot open GESTIONALE FTP connection",["folder"=>$ftpPath]);
            return false;
        }
    }

    /**
     * Upload file to specific directory via FTP
     * @param $fileName
     * @param $filePath
     * @param $ftpPath
     * @return void
     */
    public function upload($fileName, $filePath, $ftpPath){
        try {
            // Open connection from $ftpPath
            $this->ftpOpen($ftpPath);

            // Read file content
            $content = file_get_contents($filePath);

            // Write file to FTP
            $this->mageFtp->write($fileName,$content);

            // Close ftp connection
            $this->mageFtp->close();

            $this->logger->info("[FTP-SEND-FILE] - Sent file {$fileName} to remote path '{$ftpPath}'",["method"=>__METHOD__]);
        }catch (\Exception $e){
            $this->logger->critical("[FTP-SEND-FILE] - Error sending file via FTP",["exception"=>$e->getMessage(),"method"=>__METHOD__]);
        }
    }


    /**
     * Check if semaphore file exists
     * @param $ftpPath
     * @return bool
     */
    public function semaphoreExists($ftpPath){
        // Open Ftp Connection
        if(!$this->ftpOpen($ftpPath)){
            return false;
        }

        // Check if semaphore file exists
        try{
            $this->mageFtp->read(self::SEMAPHORE_FILE,self::FTP_IMPORT_PATH);
            $this->mageFtp->close();
            return true;
        }catch (\Exception $e){
            return false;
        }
    }

    /**
     * Read filename from path via FTP
     * @param $filename
     * @param $path
     * @param bool $delete
     * @return string $result
     */
    public function read($filename, $path, $delete=false){
        // Open connection from $path
        $this->ftpOpen($path);

        // Read file requested
        $result = $this->mageFtp->read($filename);

        // If requested, delete file
        if($delete){
            $this->mageFtp->rm($filename);
        }

        // Close connection
        $this->mageFtp->close();
        return $result;

    }

    /**
     * List FTP files and directory into path requested
     * @param $path
     * @return array
     */
    public function list($path){
        $this->ftpOpen($path);
        $result = $this->mageFtp->ls();
        $this->mageFtp->close();
        return $result;
    }


    /**
     * Create an empty file for semaphore
     * @param $semaphorePath
     * @param $ftpPath
     */
    public function writeSemaphore($semaphorePath, $ftpPath){
        try {
            // Open connection from $ftpPath
            $this->ftpOpen($ftpPath);

            // Read semaphore file
            $content = file_get_contents($semaphorePath);

            $this->mageFtp->write(self::SEMAPHORE_FILE,$content);

            $this->logger->info("[FTP-SEMAPHORE] - Wrote semaphore to '{$ftpPath}' directory via FTP");

            // Close ftp connection
            $this->mageFtp->close();
        }catch (\Exception $e){
            $this->logger->error("[FTP-SEMAPHORE] - Error while writing semaphore. " . $e->getMessage());
        }
    }

    /**
     * Delete semaphore file into remote directory
     * @param $ftpPath
     */
    public function semaphoreDelete($ftpPath){
        try{
            // Open connection from $ftpPath
            $this->ftpOpen($ftpPath);

            // Delete semaphore file
            $this->mageFtp->rm(self::SEMAPHORE_FILE);

            $this->logger->info("[FTP-SEMAPHORE] - Deleted semaphore from '{$ftpPath}' directory via FTP");

            // Close ftp connection
            $this->mageFtp->close();
        }catch (\Exception $e){
            $this->logger->error("[FTP-SEMAPHORE] - Error while deleting semaphore. " . $e->getMessage());
        }
    }
}