<?php
/**
 * Created by Simone Monterubbiano.
 * Email: s.monterubbiano@drop.it
 * Date: 13/07/2018
 * File name: Import.php
 * Project: JustEat
 */

namespace Drop\Gestionale\Model\Shipping;

class Import{

    // constants
    const FTP_SHIPPING_PATH = "EXPORT";
    const LOCAL_SHIPPING_FILE_PATH = "pub/gestionale/shipping";
    private $shippingFilesName =[];

    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $dir;
    /**
     * @var \Magento\Framework\Xml\Parser
     */
    private $xmlParser;
    /**
     * @var \Magento\Sales\Model\Order\Shipment\TrackFactory
     */
    private $trackFactory;
    /**
     * @var \Magento\Sales\Model\Order
     */
    private $order;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;
    /**
     * @var \Drop\Gestionale\Model\GestionaleFtp
     */
    private $twsFtp;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $file;
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Magento\Sales\Model\Convert\Order
     */
    private $orderConvert;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\Sales\Model\OrderRepository
     */
    private $orderRepository;

    /**
     * Import constructor.
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     * @param \Magento\Framework\Xml\Parser $xmlParser
     * @param \Magento\Sales\Model\Order\Shipment\TrackFactory $trackFactory
     * @param \Magento\Sales\Model\Order $order
     * @param \Magento\Sales\Model\OrderRepository $orderRepository
     * @param \Magento\Sales\Model\Convert\Order $orderConvert
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     * @param \Drop\Gestionale\Model\GestionaleFtp $twsFtp
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Helper\Data $helper
     */
    public function __construct(
        // Magento
        \Magento\Framework\Filesystem\DirectoryList $dir,
        \Magento\Framework\Xml\Parser $xmlParser,
        \Magento\Sales\Model\Order\Shipment\TrackFactory $trackFactory,
        \Magento\Sales\Model\Order $order,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Sales\Model\Convert\Order $orderConvert,
        \Magento\Framework\Filesystem\Io\File $file,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\Message\ManagerInterface $messageManager,

        // Drop
        \Drop\Gestionale\Model\GestionaleFtp $twsFtp,
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Helper\Data $helper
    ) {

        $this->dir = $dir;
        $this->xmlParser = $xmlParser;
        $this->trackFactory = $trackFactory;
        $this->order = $order;
        $this->date = $date;
        $this->twsFtp = $twsFtp;
        $this->file = $file;
        $this->logger = $logger;
        $this->orderConvert = $orderConvert;
        $this->messageManager = $messageManager;
        $this->helper = $helper;
        $this->orderRepository = $orderRepository;
    }

    public function import() {
        // Check if GESTIONALE enabled
        if(!$this->helper->isGestionaleEnabled()){
            return false;
        }

        // Check semaphore exists, else return
        if(!$this->twsFtp->semaphoreExists(self::FTP_SHIPPING_PATH)){
            $this->logger->info("[SHIPPING-IMPORT] - Semaphore TransferOk not exists, exit",["method"=>__METHOD__]);
            $this->messageManager->addWarningMessage("Cannot import, semaphore not exists");
            return false;
        }

        // Read shipping file founded on GESTIONALE ftp server, set tracking info and save on local path
        foreach($this->getShippingFiles() as $file):
            $shippingContent = $this->twsFtp->read($file,self::FTP_SHIPPING_PATH, false);

            // Save XML to local path for log
            $localFilePath = $this->dir->getRoot() . DIRECTORY_SEPARATOR . self::LOCAL_SHIPPING_FILE_PATH . DIRECTORY_SEPARATOR . $file;
            $this->file->write( $localFilePath, $shippingContent, 0666 );
            $this->logger->info("[SHIPPING-IMPORT] - Saved XML file locally",["file"=>$file,"method"=>__METHOD__]);

            // Get shipping info from content XML
            $shippingInfo = $this->getShippingInfo($shippingContent);

            // Create shipping
            if($this->createShipping($shippingInfo)) {

                // Add tracking info to shipping
                $this->addTrackingNumber($shippingInfo);
            }
        endforeach;

        // Remove semaphore, so GESTIONALE can write again
        $this->twsFtp->semaphoreDelete(self::FTP_SHIPPING_PATH);
    }

    private function addTrackingNumber($shippingInfo) {
        try {
            $order = $this->orderRepository->get($shippingInfo["order_id"]);

            $shipments = $order->getShipmentsCollection();
            $data = array(
                'carrier_code' => $shippingInfo["carrier"],
                'title' => $shippingInfo["carrier"],
                'number' => $shippingInfo["track_number"], // Replace with your tracking number
            );
            foreach($shipments as $shipment):
                $track = $this->trackFactory->create()->addData($data);
                $shipment->addTrack($track)->save();
            endforeach;
        } catch (\Exception $e) {
            $this->logger->error("[SHIPPING-IMPORT] - Error while adding tracking",["exception"=> $e->getMessage()]);
        }
    }

    /**
     * Get stock file from FTP: the file name differs every time
     * @return string Stock file content
     */
    private function getShippingFiles() {
        // List all files into FTP
        $shippingFilesName=[];
        $result = $this->twsFtp->list(self::FTP_SHIPPING_PATH);
        if(!count($result)){
            $this->logger->info("No shipping files into shipping directory",["method"=>__METHOD__]);
        }

        // Search for shipping filename
        foreach ($result as $file):
            if(strpos($file["text"],"xml")){
                $shippingFilesName[] = str_replace("./","",$file["text"]);
            }
        endforeach;

        return $shippingFilesName;
    }

    /**
     * @param $shippingContent
     * @return array
     */
    private function getShippingInfo($shippingContent){
        $xml=simplexml_load_string($shippingContent);
        $shippingInfo = [
            "order_id" => (string)$xml->Reference,
            "date" => (string)$xml->Body->MessageParts->MJShipmentInvoices->MJInboundInvoice->Ship_Date,
            "carrier" => (string)$xml->Body->MessageParts->MJShipmentInvoices->MJInboundInvoice->ShipperDescr,
            "weight" => (string)$xml->Body->MessageParts->MJShipmentInvoices->MJInboundInvoice->Weight,
            "track_number" => (string)$xml->Body->MessageParts->MJShipmentInvoices->MJInboundInvoice->MJInboundCartonHeader->Track_No,
        ];

        return $shippingInfo;
    }

    private function createShipping($shippingInfo){

        $order = $this->orderRepository->get($shippingInfo["order_id"]);

        $this->logger->info("[SHIPPING-IMPORT] - Starting to create shipping",["order"=>$order->getIncrementId(),"method"=>__METHOD__]);

        // Check if order can be shipped or has already shipped
        if (!$order->canShip()) {
            $this->logger->error("[SHIPPING-IMPORT] - Order cannot be shipped, exit",["order"=>$order->getIncrementId(),"method"=>__METHOD__]);
            $this->messageManager->addComplexErrorMessage("Order #{$order->getIncrementId()} cannot be shipped");
            return false;
        }

        // Initialize the order shipment object
        $shipment = $this->orderConvert->toShipment($order);

        // Loop through order items
        foreach ($order->getAllItems() as $orderItem) {

            // Check if order item has qty to ship or is virtual
//            if (! $orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
//                continue;
//            }

            $qtyShipped = $orderItem->getQtyToShip();

            // Create shipment item with qty
            $shipmentItem = $this->orderConvert->itemToShipmentItem($orderItem)->setQty($qtyShipped);

            // Add shipment item to shipment
            $shipment->addItem($shipmentItem);
        }

        // Register shipment
        $shipment->register();

        $shipment->getOrder()->setIsInProcess(true);

        try {
            // Save created shipment and order
            $shipment->save();
            $shipment->getOrder()->save();

            // Send email
            //$this->_objectManager->create('Magento\Shipping\Model\ShipmentNotifier')->notify($shipment);

            $shipment->save();
            $this->logger->info("[SHIPPING-IMPORT] - Shipping created,",["order"=>$order->getIncrementId(),"method"=>__METHOD__]);
        } catch (\Exception $e) {
            $this->logger->error("[SHIPPING-IMPORT] - Error while creating shipping,",["exception"=>$e->getMessage(),"order"=>$order->getIncrementId(),"method"=>__METHOD__]);
            $this->messageManager->addComplexErrorMessage("Cannot create shipping",["exception"=>$e->getMessage()]);
            return false;
        }
        return true;
    }
}
