<?php

/*
 * Description: Stock.php
 * @date: 22-mag-2018
 * @author Simone Monterubbiano <s.monterubbiano@drop.it>
 */

namespace Drop\Gestionale\Model;

class Stock {

    const LOG_TEMPLATE_EMAIL_NAME = "drop_tws_stock_log";

    private $logMessageEmail;
    private $stockFileName;

    /**
     * @var GestionaleFtp
     */
    private $twsFtp;
    /**
     * @var \Magento\CatalogInventory\Api\StockRegistryInterface
     */
    private $stockRegistry;
    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $dir;
    /**
     * @var \Magento\Framework\File\Csv
     */
    private $csv;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var Email
     */
    private $email;
    /**
     * @var \Drop\Gestionale\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Gestionale\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $file;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory
     */
    private $productCollectionFactory;


    /**
     * Stock constructor.
     * @param \Magento\CatalogInventory\Api\StockRegistryInterface $stockRegistry
     * @param \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollectionFactory
     * @param \Magento\Framework\Filesystem\DirectoryList $dir
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\File\Csv $csv
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     * @param Email $email
     * @param \Drop\Gestionale\Logger\Logger $logger
     * @param \Drop\Gestionale\Helper\Data $helper
     * @param GestionaleFtp $twsFtp
     */
    public function __construct(
        // Magento
        \Magento\CatalogInventory\Api\StockRegistryInterface $stockRegistry,
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollectionFactory,
        \Magento\Framework\Filesystem\DirectoryList $dir,
        \Magento\Framework\Filesystem\Io\File $file,
        \Magento\Framework\File\Csv $csv, 
        \Magento\Framework\Message\ManagerInterface $messageManager,

        // Drop
        \Drop\Gestionale\Model\Email $email,
        \Drop\Gestionale\Logger\Logger $logger,
        \Drop\Gestionale\Helper\Data $helper,
        \Drop\Gestionale\Model\GestionaleFtp $twsFtp
    ){

        $this->stockRegistry = $stockRegistry;
        $this->dir = $dir;
        $this->csv = $csv;
        $this->messageManager = $messageManager;
        $this->email = $email;
        $this->logger = $logger;
        $this->helper = $helper;
        $this->twsFtp = $twsFtp;
        $this->file = $file;
        $this->productCollectionFactory = $productCollectionFactory;
    }

    /**
     * Main method for update stock
     * @throws \Exception
     */
    public function import() {
        // Check if GESTIONALE enabled
        if(!$this->helper->isGestionaleEnabled()){
            return false;
        }

        $time_start = microtime(true);

        // Load stock file content from FTP
        $stockFileContent = $this->getStockFileContent();
        if (empty($stockFileContent)) {
            // Log error to file
            $this->logger->error("[STOCK] - No stock file available on GESTIONALE remote folder",["method"=>__METHOD__]);

            // Log error to session admin
            $this->messageManager->addErrorMessage("No stock file available on GESTIONALE remote folder");

            // Log to email
            $this->logMessageEmail .= "No stock file available on GESTIONALE remote folder.<br/>";
            $this->sendLogEmail();
            return;
        }

        $localFilePath = $this->dir->getRoot() . DIRECTORY_SEPARATOR . $this->helper->getLocalStockPath() . DIRECTORY_SEPARATOR . $this->stockFileName;
        try {
            // Save content stock file to local file
            $this->file->write( $localFilePath, $stockFileContent, 0666 );

            // Log message email
            $this->logMessageEmail .= "Saved stock file from remote to local directory.<br/>";

            // Log message to file
            $this->logger->info("[STOCK] - Saved stock file from remote to local directory",["method"=>__METHOD__]);

            // Log message to session admin
            $this->messageManager->addSuccessMessage("Saved stock file from remote to local directory");
        }catch(\Exception $e){
            // Log message to file
            $this->logger->error("[STOCK] [EXCEPTION] - Cannot save stock file into local directory",["method"=>__METHOD__]);

            // Log warning message to session admin
            $this->messageManager->addWarningMessage("Cannot save stock file into local directory");

            // Log message email
            $this->logMessageEmail .= "Cannot save stock file into local directory.<br/>";
        }

        // Read stock file into array (shift first element)
        $stock = $this->csv->getData($localFilePath);
        array_shift($stock);

        // Prepare stock array
        $arrayStock = $this->prepareStockArray($stock);

        // If stock array is empty, return
        if(!count($arrayStock)){
            // Log error to file
            $this->logger->info("[STOCK] - Stock file is empty, exit",["method"=>__METHOD__]);

            // Log error message to session admin
            $this->messageManager->addWarningMessage("Stock file is empty, exit");

            // Send log error via email
            $this->logMessageEmail .= "Stock file is empty<br/>";
            $this->sendLogEmail();
            return;
        }

        // Product collection for mapping barcode/sku
        $arrayMapBarcodeSku = [];
        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect(["sku","barcode"]);
        foreach($collection as $product):
            if(empty($product->getBarcode())){
                continue;
            }
            $arrayMapBarcodeSku[trim($product->getBarcode())]=trim($product->getSku());
        endforeach;

        // Product collection for mapping sku/barcode
        $arrayMapSkuBarcode = [];
        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect(["sku","barcode"]);
        foreach($collection as $product):
            if(empty($product->getBarcode())){
                continue;
            }
            $arrayMapSkuBarcode[$product->getSku()]=trim(trim($product->getBarcode()));
        endforeach;

        $this->logger->info("[STOCK] - Begin stock update",["products"=>count($stock),"method"=>__METHOD__]);
        $this->logMessageEmail .= "Begin stock update: file: {$this->stockFileName} (" . count($stock) . " products)<br/><br/>";

        // Get product kit info
        $this->logMessageEmail .= "<strong>Begin bundle stock update.</strong><br/><br/>";
        $arrayKit = $this->getKit();

        // Cycle array kit for getting each product quantity in the kit, if one of them is zero, set bundle out of stock else set in stock
        foreach($arrayKit as $bundle=>$kitProducts):
            $inStock = false;
            foreach ($kitProducts as $sku) {

                // Check if sku of the bundle is available
                if(!isset($arrayMapSkuBarcode[$sku])){
                    $this->logMessageEmail .= "Bundle {$bundle}: {$sku} non presente <br/>";
                    continue;
                }

                // Get barcode from sku
                $barcode = $arrayMapSkuBarcode[$sku];

                if($arrayStock[$barcode]==0){
                    $inStock = false;
                }else{
                    $inStock = true;
                }
            }
            if(!$inStock){
                $this->logMessageEmail .= "Set bundle {$bundle} OUT OF STOCK<br/>";
                $this->setQty($bundle,$bundle,0);
            }else{
                $this->logMessageEmail .= "Set bundle {$bundle} IN STOCK<br/>";
                $this->setQty($bundle,$bundle,1);
            }
        endforeach;

        // Begin update stock
        $this->logMessageEmail .= "<br/><strong>Begin simple products stock update.</strong><br/><br/>";
        foreach ($arrayStock as $barcode=>$qty):
            // If product not has sku, continue
            if(!isset($arrayMapBarcodeSku[$barcode])){
                continue;
            }
            // Set product quantity
            $this->setQty($arrayMapBarcodeSku[$barcode],$barcode, $qty);
        endforeach;

        $time_end = microtime(true);
        $time = $time_end - $time_start;

        // Debug
//        $this->logMessageEmail;
//        die(__METHOD__);

        // Send log email
        $this->logMessageEmail .= "<br/>Update stock finished in {$time} seconds<br/>";
        $this->sendLogEmail();

        // Log message session admin
        $this->messageManager->addSuccessMessage(__("Updated stock correctly, executed in {$time} seconds."));

        // Log to file
        $this->logger->info("[STOCK] - Update stock finished in {$time} seconds",["products"=>count($stock),"method"=>__METHOD__]);
    }
    
    /**
     * Prepare array to pass to importer
     * @param array $stock
     * @return array
     */
    public function prepareStockArray($stock){
        $array=[];
        foreach($stock as $row):
//            $array[] = array(
//                "barcode" => $row[0],
//                "qty" => str_replace('"',"",$row[1])
//            );
            $array[$row[0]] = str_replace('"',"",$row[1]);
        endforeach;
        return $array;
    }

    /**
     * Set product qty by sku
     * @param string $sku
     * @param $barcode
     * @param integer $qty
     */
    private function setQty($sku, $barcode, $qty) {
        try {
            $stockItem = $this->stockRegistry->getStockItemBySku($sku);
            $stockItem->setQty($qty);
            $stockItem->setIsInStock((bool) $qty);
            $this->stockRegistry->updateStockItemBySku($sku, $stockItem);
            $this->logMessageEmail .= "Sku <b>{$sku}</b> <small>Barcode: ({$barcode})</small> - Qty: {$qty}<br/>";
        } catch (\Exception $e) {
            $this->logger->error("[STOCK] - [EXCEPTION]:".$e->getMessage(),["method"=>__METHOD__]);
        }
    }

    /**
     * Get stock file from FTP: the file name differs every time
     * @return string Stock file content
     */
    public function getStockFileContent() {
        // List all files into FTP
        $result = $this->twsFtp->list($this->helper->getRemoteFtpStockPath());
        if(!count($result)){
            $this->logger->debug("[STOCK] - No stock file into ftp directory",["method"=>__METHOD__]);
        }

        // Search for stock filename
        foreach ($result as $file):
            if(strpos($file["text"],"csv")){
                $this->stockFileName = str_replace("./","",$file["text"]);
            }
        endforeach;

        // Read stock file founded on GESTIONALE ftp server
        return $this->twsFtp->read($this->stockFileName,$this->helper->getRemoteFtpStockPath(), false);
    }

    /**
     * Return array Kit
     * @return array
     * @throws \Exception
     */
    private function getKit(){
        // Update bundle stock status based on associated product stock status
        $kitFilePath = $this->dir->getRoot() . DIRECTORY_SEPARATOR . "pub/justeat/kit.csv";
        $kitContent = $this->csv->getData($kitFilePath);
        $arrayKit=[];
        foreach($kitContent as $kit):
            $productKits = explode(",",$kit[1]);
            $productKitSingle = explode("+",$productKits[0]);
            $arrayKit[$kit[0]] = $productKitSingle;
        endforeach;
        return $arrayKit;
    }

    /**
     * Send log email
     */
    private function sendLogEmail(){
        $tos = explode(";",$this->helper->getStockLogEmail());

        $this->email->send(
            $tos,
            "drop_tws_stock_log",
            ['title' => "Stock update", 'message' => $this->logMessageEmail]);
    }
}