<?php
/**
 * Apply
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Geolocation\Model;

use \Magento\Store\Model\ScopeInterface;

/**
 * Class Apply
 *
 * @package Drop\Geolocation\Model
 */
class Apply
{
    /**
     * @var Session
     */
    protected $session;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var \Drop\Geolocation\Logger\Logger
     */
    private $logger;
    /**
     * @var \Magento\Directory\Model\AllowedCountries
     */
    private $allowedCountry;
    /**
     * @var \Magento\Framework\UrlInterface
     */
    private $urlInterface;
    /**
     * @var \Magento\UrlRewrite\Model\UrlRewriteFactory
     */
    private $urlRewriteFactory;
    /**
     * @var \Drop\Geolocation\Helper\Data
     */
    private $helper;
    /**
     * @var \Drop\Geolocation\Helper\Apply
     */
    private $applyHelper;
    /**
     * @var \Magento\Framework\App\Response\RedirectInterface
     */
    private $redirect;

    /**
     * Apply constructor
     *
     * @param Session $session
     * @param \Magento\Directory\Model\AllowedCountries $allowedCountry
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\UrlInterface $urlInterface
     * @param \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteFactory
     * @param \Magento\Framework\App\Response\RedirectInterface $redirect
     * @param \Drop\Geolocation\Helper\Data $helper
     * @param \Drop\Geolocation\Helper\Apply $applyHelper
     * @param \Drop\Geolocation\Logger\Logger $logger
     */
    public function __construct(
        Session $session,
        \Magento\Directory\Model\AllowedCountries $allowedCountry,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Framework\UrlInterface $urlInterface,
        \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteFactory,
        \Magento\Framework\App\Response\RedirectInterface $redirect,
        \Drop\Geolocation\Helper\Data $helper,
        \Drop\Geolocation\Helper\Apply $applyHelper,
        \Drop\Geolocation\Logger\Logger $logger
    ) {
        $this->session = $session;
        $this->allowedCountry = $allowedCountry;
        $this->storeManager = $storeManager;
        $this->urlInterface = $urlInterface;
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->redirect = $redirect;
        $this->helper = $helper;
        $this->applyHelper = $applyHelper;
        $this->logger = $logger;
    }

    /**
     * Calculate the right destination url
     *
     * @param $destinationCountryId
     * @param $destinationLocale
     * @param $requestedUrl
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getUrlToRightStoreView($destinationCountryId, $destinationLocale, $requestedUrl)
    {
        if(empty($destinationCountryId) || empty($destinationLocale)) {
            $this->logger->error(__METHOD__ . ' - Empty CountryId/Locale.');
            return $this->urlInterface->getUrl(\Drop\Geolocation\Helper\Data::FALLBACK_PAGE_URL);
        }

        $destinationStore = $this->getStoreByCountryAndLocale($destinationCountryId, $destinationLocale);
        if(empty($destinationStore)) {
            //Disabled Country
            return $this->urlInterface->getUrl(\Drop\Geolocation\Helper\Data::FALLBACK_PAGE_URL);
        }

        if($this->helper->getIsGeolocationOrStoreSwitcherUrl($requestedUrl)) {
            return $this->helper->getBaseUrl() . $destinationStore->getCode() . '/';
        }

        if($this->applyHelper->getStoreCodeIsOnUrl($requestedUrl) && ($destinationStore->getCode() == $this->applyHelper->getStoreCodeFromUrl($requestedUrl))) {
            //if store_code is in url and is the same of destination
            $this->session
                ->setCountryId($destinationCountryId)
                ->setLocale($destinationLocale)
                ->setUserIsManuallyGeolocated(true);
            return false;
        }

        if(!$this->applyHelper->getStoreCodeIsOnUrl($requestedUrl) &&
            ($this->applyHelper->getCurrentStoreCode() == $this->applyHelper->getDefaultStoreCode()) &&
            ($destinationStore->getCode() == $this->applyHelper->getDefaultStoreCode())
        ) {
            //if store_code is not in url but current store code and destination store code are the default store code (like home page and italian geolocation)
            $this->session
                ->setCountryId($destinationCountryId)
                ->setLocale($destinationLocale)
                ->setUserIsManuallyGeolocated(true);
            return false;
        }

        try {
            $destinationPath = $this->getDestinationPath($destinationStore, $requestedUrl);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $destinationPath = '/';
        }

        $this->session
            ->setCountryId($destinationCountryId)
            ->setLocale($destinationLocale)
            ->setUserIsManuallyGeolocated(true);

        return $this->helper->getBaseUrl() . $this->urlInterface->getRedirectUrl($destinationPath);
    }

    /**
     * Get store codes by requested country_id
     * @param $countryId
     * @param $locale
     * @return \Magento\Store\Api\Data\StoreInterface
     */
    public function getStoreByCountryAndLocale($countryId, $locale)
    {
        $stores = $this->applyHelper->getStoreList();
        $rightStore = '';
        $foundEnglishCountryId = false;

        foreach ($stores as $store):
            $allowedCountries = $this->allowedCountry->getCountriesFromConfig(ScopeInterface::SCOPE_STORES, $store->getId());

            if(in_array($countryId, $allowedCountries) && ($locale == $this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store))) {
                //Found country and browser locale. EXIT
                $rightStore = $store;
                break;
            }
            if(in_array($countryId, $allowedCountries) && ($this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store) == \Drop\Geolocation\Helper\Data::DEFAULT_MAGENTO_LANGUAGE)) {
                // IF Found country but not browser locale, used en_US and locale
                $rightStore = $store;
                $foundEnglishCountryId = true;
                continue;
            }

            if(!$foundEnglishCountryId && in_array($countryId, $allowedCountries)) {
                // IF not found english, search website only for the right country id
                $rightStore = $store;
            }
        endforeach;

        return $rightStore;
    }

    /**
     * Get request_path for the requested store_code
     *
     * @param $store
     * @param $requestedUrl
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getDestinationPath($store, $requestedUrl)
    {
        $queryStrings = $this->getUrlQueryString($requestedUrl);
        $requestPath = $this->getUrlRequestPath($requestedUrl);

        if(empty($requestPath)) {
            return $store->getCode() . '/' . $queryStrings;
        }

        $translatedPath = $this->getUrlStoreTranslation($requestPath, $this->storeManager->getStore(), $store);

        if(strpos($translatedPath, 'checkout') === 0) {
            //Force redirect to cart if url calculation is in checkout
            $translatedPath = 'checkout/cart/';
        }

        return $store->getCode() . '/' . $translatedPath . $queryStrings;
    }

    /**
     * Calc url in destination store
     *
     * @param $requestedPath
     * @param $originalStore
     * @param $destinationStore
     * @return string
     *
     * @todo Missing Connect with translations funtionality
     */
    public function getUrlStoreTranslation($requestedPath, $originalStore, $destinationStore)
    {
        try {
            $rewriteCollection = $this->urlRewriteFactory->create()->getCollection();
            $urlRewrite = $rewriteCollection
                ->addFieldToFilter('request_path', ['eq' => $requestedPath])
//                ->addFieldToFilter('store_id', ['eq' => $originalStore->getStoreId()])
                ->addFieldToFilter('redirect_type', ['eq' => 0])
                ->getFirstItem();
            $targetPath = $urlRewrite->getTargetPath();
            $entityType = $urlRewrite->getEntityType();
        } catch(\Exception $e) {
            $this->logger->error($e->getMessage());
            $targetPath = $entityType = '';
        }

        $searchField = 'target_path';
        if($entityType == 'cms-page') {
            //@todo: This is a workaround. Missing Connect with translations funtionality
            $targetPath = $requestedPath;
            $searchField = 'request_path';
        }

        if(empty($targetPath)) {
            // If empty target path (not found on url_rewrite table) is a system url (like home, checkout, customer/account, etc.)
            return $requestedPath;
        }

        try {
            $rewriteCollection = $this->urlRewriteFactory->create()->getCollection();
            $translatedPath = $rewriteCollection
                ->addFieldToFilter('store_id', ['eq' => $destinationStore->getStoreId()])
                ->addFieldToFilter($searchField, ['eq' => $targetPath])
                ->addFieldToFilter('redirect_type', ['eq' => 0])
                ->getFirstItem()
                ->getRequestPath();
        } catch (\Exception $e){
            $this->logger->error($e->getMessage());
            $translatedPath = '';
        }

        return $translatedPath;
    }

    /**
     * Get the clean url by removing query strings, baseurl and store code
     *
     * @param $url
     * @return bool|string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getUrlRequestPath($url)
    {
        if (empty($url) || $url === '/') {
            return '';
        }

        $url = $this->applyHelper->removeQueryStringFromUrl($url);
        $url = $this->applyHelper->removeBaseUrlFromUrl($url);
        $url = $this->applyHelper->removeStoreCodeFromUrl($url);
        if (substr($url,0,1) == '/') {
            $url = ltrim($url, '/');
        }

        return $url;
    }

    /**
     * Return a string of query strings from url
     *
     * @param $url
     * @return string
     */
    public function getUrlQueryString($url)
    {
        if(strpos($url, '?') === false) {
            return '';
        }

        $parsedQueryStrings = [];
        $parsedUrl = parse_url($url);

        if(isset($parsedUrl['query'])) {
            parse_str($parsedUrl['query'],$parsedQueryStrings);

            $sidQueryParam = \Magento\Framework\Session\SidResolverInterface::SESSION_ID_QUERY_PARAM;
            if(isset($parsedQueryStrings[$sidQueryParam])) {
                unset($parsedQueryStrings[$sidQueryParam]);
            }
        }

        if(empty($parsedQueryStrings)) {
            return '';
        }
        return '?' . http_build_query($parsedQueryStrings);
    }

}
