<?php
/**
 * Data
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Geolocation\Helper;

use \Magento\Store\Model\ScopeInterface;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{

    const DEFAULT_MAGENTO_LANGUAGE = 'en_US';
    const DEFAULT_MAGENTO_COUNTRY_ID = 'US';

    const FALLBACK_PAGE_URL = 'geolocation/index/fallback';
    const GEOLOCATION_CONTROLLER_APPLY_URL = 'geolocation/index/apply';
    const GEOLOCATION_CONTROLLER_SHIP_TO_URL = 'geolocation/index/shipto';
    const GEOLOCATION_CONTROLLER_CHANGE_URL = 'geolocation/index/change';
    const GEOLOCATION_CONTROLLER_LOCALE_URL = 'geolocation/index/locale';
    const GEOLOCATION_CONTROLLER_DIFFERENT_COUNTRY_URL = 'geolocation/index/differentcountry';
    const STORE_REDIRECT_URL = 'stores/store/redirect';

    const STORE_SWITCHER_URL = 'stores/store/switch';
    const ESI_REQUEST_URL = 'page_cache/block/esi';
    const XML_PATH_GENERAL_COUNTRY_DEFAULT     = 'general/country/default';

    const XML_PATH_GENERAL_LOCALE_CODE = 'general/locale/code';
    const XML_PATH_WEB_BASE_URL = 'web/secure/base_url';
    const ALLOWED_COUNTRIES_PATH = 'general/country/allow';

    const XML_PATH_VARNISH_CACHING = 'system/full_page_cache/caching_application';
    const XML_PATH_VARNISH_BEHAVIOR = 'geolocation/general/varnish_cache_behavior';

    /**
     * Data constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context
    ) {
        parent::__construct($context);

        $this->scopeConfig = $context->getScopeConfig();
    }

    public function getScopeConfig(
        $path,
        $scopeType = ScopeInterface::SCOPE_STORE,
        $scopeCode = null
    ) {
        return $this->scopeConfig->getValue($path, $scopeType, $scopeCode);
    }

    /**
     * Get default country identifier
     *
     * @param $scopeType
     * @param $scopeCode
     * @return string
     */
    public function getDefaultCountryId($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        $defaultCountryId = $this->getScopeConfig(self::XML_PATH_GENERAL_COUNTRY_DEFAULT, $scopeType, $scopeCode);
        if(empty($defaultCountryId)) {
            return self::DEFAULT_MAGENTO_COUNTRY_ID;
        }
        return $defaultCountryId;
    }

    /**
     * Get default locale
     *
     * @param $scopeType
     * @param $scopeCode
     * @return string
     */
    public function getDefaultLocale($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        $defaultLocale = $this->getScopeConfig(self::XML_PATH_GENERAL_LOCALE_CODE, $scopeType, $scopeCode);
        if(empty($defaultLocale)) {
            return self::DEFAULT_MAGENTO_LANGUAGE;
        }
        return $defaultLocale;
    }

    /**
     * Get Magento base url configuration
     *
     * @param $scopeType
     * @param $scopeCode
     * @return string
     */
    public function getBaseUrl($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        return $this->getScopeConfig(self::XML_PATH_WEB_BASE_URL, $scopeType, $scopeCode);
    }

    /**
     * Get admin allowed countries
     *
     * @param $scopeType
     * @param null $scopeCode
     * @return mixed
     */
    public function getGeneralCountryAllowed($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        return $this->getScopeConfig(self::ALLOWED_COUNTRIES_PATH, $scopeType, $scopeCode);
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsFallbackUrl($url)
    {
        if(strpos($url, self::FALLBACK_PAGE_URL) !== false) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsGeolocationControllerApplyUrl($url)
    {
        if(strpos($url, self::GEOLOCATION_CONTROLLER_APPLY_URL) !== false) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsGeolocationOrFallbackUrl($url)
    {
        if($this->getIsFallbackUrl($url) || $this->getIsGeolocationControllerApplyUrl($url) ) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsStoreSwitcher($url)
    {
        if(strpos($url, self::STORE_SWITCHER_URL) !== false) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsStoreRedirect($url)
    {
        if(strpos($url, self::STORE_REDIRECT_URL) !== false) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsStoreSwitcherOrRedirectUrl($url)
    {
        if($this->getIsStoreRedirect($url) || $this->getIsStoreSwitcher($url) ) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsGeolocationOrStoreSwitcherUrl($url)
    {
        if($this->getIsGeolocationOrFallbackUrl($url) || $this->getIsStoreSwitcherOrRedirectUrl($url) ) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function isESIRequest($url)
    {
        if(strpos($url, self::ESI_REQUEST_URL) !== false) {
            return true;
        }
        return false;
    }

    /**
     * @param $url
     * @return bool
     */
    public function getIsExcludedFromGeolocation($url)
    {
        if($this->getIsGeolocationOrStoreSwitcherUrl($url) || $this->isESIRequest($url)) {
            return true;
        }
        return false;
    }

    /**
     * @param string $scopeType
     * @param null $scopeCode
     * @return bool
     */
    public function getVarnishIsActive($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        if($this->getScopeConfig(self::XML_PATH_VARNISH_CACHING, $scopeType, $scopeCode) == 2) {
            return true;
        }
        return false;
    }

    /**
     * @todo Redirect in PHP are excluded only if Varnish is active FOR CURRENT PAGE
     *
     * @return bool
     */
    public function getVarnishIsActiveForUrl() {
        return $this->getVarnishIsActive();
    }

    /**
     * @param string $scopeType
     * @param null $scopeCode
     * @return mixed
     */
    public function getVarnishBehavior($scopeType = ScopeInterface::SCOPE_STORE, $scopeCode = null)
    {
        return $this->getScopeConfig(self::XML_PATH_VARNISH_BEHAVIOR, $scopeType, $scopeCode);
    }

}
