<?php
/**
 * Locale
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Geolocation\Helper;

use \Magento\Store\Model\ScopeInterface;
use \Magento\Framework\Locale\Bundle\LanguageBundle;
use \Magento\Framework\Locale\Bundle\RegionBundle;

/**
 * Class Data
 *
 * @package Drop\Geolocation\Helper\GeoIp
 */
class Locale extends \Magento\Framework\App\Helper\AbstractHelper
{

    const DEFAULT_BROWSER_LANGUAGE = 'en_US';

    /**
     * @var \Drop\Geolocation\Logger\Logger
     */
    private $logger;

    /**
     * @var \Magento\Framework\App\RequestInterface
     */
    private $request;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var Data
     */
    private $helper;

    /**
     * Locale constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param Data $helper
     * @param \Drop\Geolocation\Logger\Logger $logger
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        Data $helper,
        \Drop\Geolocation\Logger\Logger $logger
    ) {
        parent::__construct($context);

        $this->request = $context->getRequest();

        $this->helper = $helper;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
    }

    /**
     * Detect browser language
     *
     * @return mixed|string
     */
    public function getBrowserLanguage()
    {
        $httpAcceptLanguage = $this->request->getServer('HTTP_ACCEPT_LANGUAGE');
        if (!isset($httpAcceptLanguage)) {
            return self::DEFAULT_BROWSER_LANGUAGE;
        }

        $httpAcceptLangs = explode(',', $httpAcceptLanguage);
        if(empty($httpAcceptLangs)) {
            return self::DEFAULT_BROWSER_LANGUAGE;
        }
        $browserLanguages = [];
        // Format browser language to match potential store codes
        foreach ($httpAcceptLangs as $httpAcceptLang) {
            $strippedLang = explode(';', $httpAcceptLang);
            $browserLanguages[] = str_replace(
                "-","_",
                $strippedLang[0]
            );
        }

        return $browserLanguages;
    }

    /**
     * Calculate Magento locale by Browser Language
     *
     * @param $browserLanguages
     * @return string
     */
    public function getMagentoLocaleByBrowserLanguage($browserLanguages)
    {
        $stores = $this->storeManager->getStores();
        foreach ($stores as $store):
            $localeCode = $this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store);
            foreach ($browserLanguages as $browserLanguage) {
                if($browserLanguage == $localeCode) {
                    return $localeCode;
                }
            }
        endforeach;
        return $this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store);
    }

    /**
     * Get Locale By Country Id
     *
     * @param $countryId
     * @return array
     */
    public function getLocalesByCountryId($countryId) {
        $locales = [];
        $stores = $this->storeManager->getStores();
        foreach($stores as $store) {
            $allowedCountries = explode(',', (string) $this->helper->getGeneralCountryAllowed(ScopeInterface::SCOPE_STORE, $store->getStoreId()));
            if(in_array($countryId, $allowedCountries)) {
                $localeCode = $this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store->getStoreId());
                $locales[] = [
                    'value' => $localeCode,
                    'label' => $this->getLocaleLabelWithCountry($localeCode)
                ];
            }
        }
        return $locales;
    }

    /**
     * Get Locale Label by Locale code
     *
     * @param $localeCode
     * @return bool|string
     */
    public function getLocaleLabel($localeCode)
    {
        $languages = (new LanguageBundle())->get($localeCode)['Languages'];
        $language = \Locale::getPrimaryLanguage($localeCode);
        if (!$languages[$language]) {
            return false;
        }
        return ucwords($languages[$language]);
    }

    /**
     * Get Locale Label by Locale Code with Country Code
     *
     * @param $localeCode
     * @return bool|string
     */
    public function getLocaleLabelWithCountry($localeCode)
    {
        $languages = (new LanguageBundle())->get($localeCode)['Languages'];
        $countries = (new RegionBundle())->get($localeCode)['Countries'];

        $language = \Locale::getPrimaryLanguage($localeCode);
        $country = \Locale::getRegion($localeCode);
        if (!$languages[$language] || !$countries[$country]) {
            return false;
        }
        return ucwords($languages[$language]) . ' (' . $countries[$country] . ')';
    }

}
