<?php
/**
 * GeoIp
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Geolocation\Helper;

/**
 * Class Data
 *
 * @package Drop\Geolocation\Helper\GeoIp
 */
class GeoIp extends \Magento\Framework\App\Helper\AbstractHelper
{

    const XML_PATH_GEOIP_OPTIONS_DATABASE_FILE     = 'geolocation/geoip/database_file';

    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    private $directoryList;

    /**
     * @var Magento\Directory\Model\ResourceModel\Country\Collection
     */
    private $countryCollection;

    /**
     * @var \Drop\Geolocation\Logger\Logger
     */
    private $logger;

    /**
     * Countries
     *
     * @var array
     */
    protected $countries;

    /**
     * GeoIp constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     * @param \Magento\Directory\Model\ResourceModel\Country\Collection $countryCollection
     * @param \Drop\Geolocation\Logger\Logger $logger
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\Filesystem\DirectoryList $directoryList,
        \Magento\Directory\Model\ResourceModel\Country\Collection $countryCollection,
        \Drop\Geolocation\Logger\Logger $logger
    ) {
        parent::__construct($context);

        $this->scopeConfig = $context->getScopeConfig();
        $this->_remoteAddress = $context->getRemoteAddress();
        $this->directoryList = $directoryList;
        $this->countryCollection = $countryCollection;
        $this->logger = $logger;
    }

    /**
     * Get ip address
     *
     * @return string
     */
    public function getIp()
    {
        $ip = $this->_remoteAddress->getRemoteAddress();
        return ($ip) ? long2ip(ip2long($ip)) : null;
    }

    /**
     * Get record by ip address
     *
     * @param $ip
     * @return bool|string|null
     * @throws \MaxMind\Db\Reader\InvalidDatabaseException
     */
    public function getCountryIdByIp($ip)
    {
        if(!$this->isValidIp($ip)) {
            return false;
        }

        $countryId = '';
        $reader = new \GeoIp2\Database\Reader($this->getDatabaseFilePath());

        try {
            $result = $reader->city($ip);
            if(!empty($result) && !empty($result->country->isoCode)) {
                $countryId = $this->getCountryIdById($result->country->isoCode);
            }
        } catch (\Exception $e){
            $this->logger->error($e->getMessage());
        }
        
        if(empty($countryId)) {
            return false;
        }

        return $countryId;
    }

    /**
     * Get country identifier by identifier
     *
     * @param string $countryId
     * @return null|string
     */
    public function getCountryIdById($countryId)
    {
        $countryId = strtoupper($countryId);
        $countries = $this->getCountries();
        return (isset($countries[$countryId])) ? $countryId : null;
    }

    /**
     * Get countries
     *
     * @return array
     */
    public function getCountries()
    {
        if (!is_null($this->countries)) {
            return $this->countries;
        }

        /** @var \Magento\Directory\Model\Country $country */
        foreach ($this->countryCollection as $country) {
            $this->countries[strtoupper($country->getId())] = $country;
        }

        return $this->countries;
    }

    /**
     * Determine if ip is ipv4
     *
     * @param $ip
     * @return bool
     */
    protected function isValidIp($ip)
    {
        if (!$ip) { return false; }
        return (!filter_var($ip, FILTER_VALIDATE_IP) == false);
    }

    /**
     * Get database file path
     *
     * @return string
     */
    protected function getDatabaseFilePath()
    {
        $path = trim($this->scopeConfig->getValue(self::XML_PATH_GEOIP_OPTIONS_DATABASE_FILE));
        if (substr($path, 0, 1) != '/') {
            $path = $this->directoryList->getRoot() . DIRECTORY_SEPARATOR . $path;
        }
        return $path;
    }

}
