<?php
/**
 * Geolocation
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Geolocation\Block;

use \Magento\Store\Model\ScopeInterface;

/**
 * Class Geolocation
 *
 * @package Drop\Geolocation\Block\
 */
class Geolocation extends \Magento\Framework\View\Element\Template
{

    /**
     * @var Magento\Store\Model\App\Emulation
     */
    private $emulation;
    /**
     * @var Magento\Directory\Block\Data
     */
    private $directoryData;
    /**
     * @var \Magento\Framework\UrlInterface
     */
    private $url;
    /**
     * @var \Magento\Framework\App\Response\RedirectInterface
     */
    private $redirect;
    /**
     * @var \Drop\Geolocation\Model\Session
     */
    private $session;
    /**
     * @var \Drop\Geolocation\Model\Apply
     */
    private $apply;
    /**
     * @var \Magento\Framework\App\Cache\Type\Config
     */
    private $configCacheType;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Magento\Framework\Serialize\SerializerInterface
     */
    private $serializer;
    /**
     * @var \Drop\Geolocation\Helper\Data
     */
    private $helper;
    /**
     * @var \Drop\Geolocation\Helper\Locale
     */
    private $localeHelper;
    /**
     * @var \Drop\Geolocation\Helper\Apply
     */
    private $applyHelper;
    /**
     * @var \Drop\Geolocation\Logger\Logger
     */
    private $logger;

    /**
     * Geolocation constructor
     *
     * @param \Magento\Framework\View\Element\Template\Context $context
     * @param \Magento\Store\Model\App\Emulation $emulation
     * @param \Magento\Directory\Block\Data $directoryData
     * @param \Magento\Framework\UrlInterface $url
     * @param \Magento\Framework\App\Response\RedirectInterface $redirect
     * @param \Drop\Geolocation\Model\Session $session
     * @param \Drop\Geolocation\Model\Apply $apply
     * @param \Magento\Framework\App\Cache\Type\Config $configCacheType
     * @param \Drop\Geolocation\Helper\Locale $localeHelper
     * @param \Drop\Geolocation\Helper\Data $helper
     * @param \Drop\Geolocation\Helper\Apply $applyHelper
     * @param \Drop\Geolocation\Logger\Logger $logger
     * @param array $data
     */
    public function __construct(
        \Magento\Framework\View\Element\Template\Context $context,
        \Magento\Store\Model\App\Emulation $emulation,
        \Drop\Geolocation\Block\Directory\Data $directoryData,
        \Magento\Framework\UrlInterface $url,
        \Magento\Framework\App\Response\RedirectInterface $redirect,
        \Drop\Geolocation\Model\Session $session,
        \Drop\Geolocation\Model\Apply $apply,
        \Magento\Framework\App\Cache\Type\Config $configCacheType,
        \Drop\Geolocation\Helper\Locale $localeHelper,
        \Drop\Geolocation\Helper\Data $helper,
        \Drop\Geolocation\Helper\Apply $applyHelper,
        \Drop\Geolocation\Logger\Logger $logger,
        array $data = []
    )
    {
        parent::__construct($context, $data);

        $this->storeManager = $context->getStoreManager();
        $this->emulation = $emulation;
        $this->directoryData = $directoryData;
        $this->url = $url;
        $this->redirect = $redirect;
        $this->session = $session;
        $this->apply = $apply;
        $this->configCacheType = $configCacheType;
        $this->localeHelper = $localeHelper;
        $this->helper = $helper;
        $this->applyHelper = $applyHelper;
        $this->logger = $logger;
    }

    /**
     * Always return countries in en_US language
     * @return string
     */
    public function getCountryHtmlSelect()
    {
        $this->emulation->startEnvironmentEmulation($this->getEnglishStoreView(), \Magento\Framework\App\Area::AREA_FRONTEND);
        $html = $this->directoryData->getCountryHtmlSelect($this->session->getGeolocatedCountryId(), 'country_id', 'geo_country_select');
        $this->emulation->stopEnvironmentEmulation();
        return $html;
    }

    /**
     * @param null $countryId
     * @param null $defValue
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getLocaleHtmlSelect($countryId = null, $defValue = null)
    {
        \Magento\Framework\Profiler::start('TEST: ' . __METHOD__, ['group' => 'TEST', 'method' => __METHOD__]);

        if($countryId == null) {
            $countryId = $this->session->getCountryId();
        }
        if($countryId == null) {
            $countryId = $this->helper->getDefaultCountryId(ScopeInterface::SCOPE_STORE);
        }

        if ($defValue === null) {
            $defValue = $this->session->getLocale();
        }

        $cacheKey = 'DIRECTORY_LOCALE_SELECT_COUNTRY_' . $countryId;
        $cache = $this->configCacheType->load($cacheKey);
        if ($cache) {
            $options = $this->getSerializer()->unserialize($cache);
        } else {
            $options = $this->localeHelper->getLocalesByCountryId($countryId);
            $this->configCacheType->save($this->getSerializer()->serialize($options), $cacheKey);
        }

        if(empty($options)) {
            $this->logger->error("Locale not found for country {$countryId}. Probably country is disabled");
            return;
        }

        $html = $this->getLayout()->createBlock(\Magento\Framework\View\Element\Html\Select::class)
            ->setName('locale')
            ->setId('geo_locale')
            ->setTitle('Language')
            ->setValue($defValue)
            ->setOptions($options)
            ->setExtraParams('data-validate="{\'validate-select\':true}"')
            ->getHtml();

        \Magento\Framework\Profiler::stop('TEST: ' . __METHOD__);
        return $html;
    }

    /**
     * Get serializer
     *
     * @return \Magento\Framework\Serialize\SerializerInterface
     * @deprecated 100.2.0
     */
    private function getSerializer()
    {
        if ($this->serializer === null) {
            $this->serializer = \Magento\Framework\App\ObjectManager::getInstance()
                ->get(\Magento\Framework\Serialize\SerializerInterface::class);
        }
        return $this->serializer;
    }

    /**
     * @return int
     */
    public function getEnglishStoreView() {
        $stores = $this->storeManager->getStores();
        foreach ($stores as $store):
            if($this->helper->getDefaultLocale(ScopeInterface::SCOPE_STORE, $store) == \Drop\Geolocation\Helper\Data::DEFAULT_MAGENTO_LANGUAGE) {
                return $store->getId();
            }
        endforeach;
        return 1;
    }

    /**
     * @return string
     */
    public function getApplyUrl() {
        return $this->getUrl(\Drop\Geolocation\Helper\Data::GEOLOCATION_CONTROLLER_APPLY_URL);
    }

    /**
     * @return string
     */
    public function getShipToUrl() {
        return $this->getUrl(\Drop\Geolocation\Helper\Data::GEOLOCATION_CONTROLLER_SHIP_TO_URL);
    }

    /**
     * @return string
     */
    public function getChangeUrl() {
        return $this->getUrl(\Drop\Geolocation\Helper\Data::GEOLOCATION_CONTROLLER_CHANGE_URL);
    }

    /**
     * @return string
     */
    public function getLocaleUrl() {
        return $this->getUrl(\Drop\Geolocation\Helper\Data::GEOLOCATION_CONTROLLER_LOCALE_URL);
    }

    /**
     * @return string
     */
    public function getDifferentCountryUrl() {
        return $this->getUrl(\Drop\Geolocation\Helper\Data::GEOLOCATION_CONTROLLER_DIFFERENT_COUNTRY_URL);
    }

    /**
     * Not check to Varnish because popup need to be opened also if Varnish is disabled.
     *
     * @return bool
     */
    public function getOpenPopupOnDifferentCountry() {
        if($this->helper->getVarnishBehavior() == \Drop\Geolocation\Model\Config\Source\VarnishBehavior::BEHAVIOR_POPUP) {
            return true;
        }
        return 0;
    }

    /**
     * @return bool
     */
    public function getUserIsManuallyGeolocated() {
        $currentUrl = $this->url->getCurrentUrl();
        if(!$this->applyHelper->getStoreCodeIsOnUrl($currentUrl)) {
            return 0;
        }
        return $this->session->getUserIsManuallyGeolocated();
    }

    /**
     * @return bool
     */
    public function getRedirectIsNeeded() {
        $currentUrl = $this->url->getCurrentUrl();
        if(!$this->applyHelper->getStoreCodeIsOnUrl($currentUrl)) {
            return true;
        }
        return 0;
    }

    public function getIsFallbackUrl() {
        $currentUrl = $this->url->getCurrentUrl();
        if($this->helper->getIsFallbackUrl($currentUrl)) {
            return true;
        }
        return false;
    }

}
